import argparse
import json
import logging
import types
import os
from collections.abc import Iterable

import osmnx as ox
from yt.wrapper import YtClient



def make_list_if_not_already(x):
    if not isinstance(x, Iterable) or isinstance(x, str):
        return [x]
    else:
        return x


def extract_roads(graph, ignore_road_types=()):
    """For each road in osmnx graph yield lat, lon and id of its legs.

    Parameters
    ----------
    graph : networkx.MultiDiGraph
        osmnx graph
    ignore_road_types : Tuple[str]
        tuple of road types to ignore
        see https://wiki.openstreetmap.org/wiki/Key:highway for the full list

    Returns
    -------
    List[Dict[str, Any]] with keys 'lat', 'lon' and 'osmid'.
    """
    edges_gdfs = ox.graph_to_gdfs(graph, edges=True, nodes=False)
    all_coords = []
    for index, row in edges_gdfs[['geometry', 'osmid', 'highway']].iterrows():
        row['osmid'] = make_list_if_not_already(row['osmid'])
        row['highway'] = make_list_if_not_already(row['highway'])
        if any(road_type in row['highway'] for road_type in ignore_road_types):
            continue
        x, y = row['geometry'].coords.xy
        coords = list(zip(list(y), list(x)))
        for osmid in row['osmid']:
            ext_coords = {}
            for c in coords:
                ext_coords['lat'] = c[0]
                ext_coords['lon'] = c[1]
                ext_coords['osmid'] = osmid
            all_coords.append(ext_coords)
    return all_coords


def main():
    parser = argparse.ArgumentParser(description='Process some integers.')
    parser.add_argument(
        '--output-table',
        default='//home/maps/users/tswr/osm/roads', required=True)
    parser.add_argument('--north', type=float, required=True)
    parser.add_argument('--south', type=float, required=True)
    parser.add_argument('--east', type=float, required=True)
    parser.add_argument('--west', type=float, required=True)
    parser.add_argument('--ignore-road-types', required=True)
    args = parser.parse_args()

    logging.basicConfig(format='%(asctime)s %(levelname)s %(message)s')
    logger = logging.getLogger('upload_roads_from_osm_to_yt')
    logger.setLevel(logging.INFO)

    logger.info('Downloading graph')
    graph = ox.graph_from_bbox(args.north, args.south, args.east, args.west,
                               network_type='drive')
    logger.info('Extracting roads')
    ignore = args.ignore_road_types.split(',') if args.ignore_road_types else ()
    all_coords = extract_roads(graph, ignore)

    logger.info('Uploading results to %s', args.output_table)
    client = YtClient(proxy="hahn")
    client.write_table(args.output_table, all_coords)


if __name__ == '__main__':
    main()
