import os
import logging
import shutil
import tarfile
import glob

from sandbox import sdk2
from sandbox import common
from sandbox.sdk2.helpers import subprocess
import sandbox.common.types.task as ctt

USER_NAME = 'builder'
USER_HOME = '/home/' + USER_NAME
BUILD_DIR = 'build'
SANEBUILD_LOG = 'build.log'

YAV_SECRET_DIR = str(os.path.join(USER_HOME, 'yav_secret'))
YAV_SECRET_DEB_NAME = 'yandex-maps-garden-yav-secret.deb'
YAV_SECRET_DEB_PATH = str(os.path.join(YAV_SECRET_DIR,
                                       'genfiles',
                                       YAV_SECRET_DEB_NAME))

YAV_MAKEFILE = \
    """package-name := yandex-maps-garden-yav-secret

SRCPKG := $(package-name)
PKG := $(package-name)

INSTALL_DIR = /etc/yandex/maps/garden/secrets.d

files = $(foreach folder,$(1),$(INSTALL_DIR)/$(folder):$(folder)/$(2))

$(package-name): FILES = $(INSTALL_DIR)/yav_secret.json

include /usr/share/yandex/maps/build/build.mk
"""

YAV_CHANGELOG = \
    """PACKAGE (1.0.0-0) unstable; urgency=medium

  * Init

  -- Nobody <nobody@yandex-team.ru>  Fri, 28 Sep 2018 11:01:06 +0300

"""


def run_subprocess(command, cwd, env=None):
    child = subprocess.Popen(
        command,
        stdout=subprocess.PIPE,
        stderr=subprocess.STDOUT,
        cwd=cwd,
        shell=True,
        env=env
    )

    out, _ = child.communicate()
    if child.returncode == 0:
        logging.debug('Finished successfully:\n' + out)
    else:
        logging.error('Task failed:\n' + out)
        raise common.errors.TaskFailure(
            'Subprocess [{}] failed:\n'.format(command)
        )


def run_sane_build_test(path, user, platform, stage, dependencies):
    debs = ''
    if dependencies:
        debs = '+' + ' +'.join(dependencies)

    run_subprocess(
        "sudo su - {user} -c '"
        "cd {path} && "
        "sudo sanebuild -L {logfile} --{platform} --{stage} {debs} -- test"
        "'".format(
            user=user,
            path=path,
            logfile=SANEBUILD_LOG,
            platform=platform,
            stage=stage,
            debs=debs
        ),
        path
    )

    run_subprocess(
        "echo 'Sanebuild log:' && "
        "cat {}".format(str(os.path.join(path, SANEBUILD_LOG))),
        path)


def mount_dir(dir_path, mount_point):
    run_subprocess(
        'sudo mount --bind {dir_path} {mount_point}'.format(
            dir_path=dir_path,
            mount_point=mount_point
        ),
        dir_path
    )


def make_yav_secret_deb(path, secret):
    if not os.path.exists(path):
        os.mkdir(path)

    with open(str(os.path.join(path, 'Makefile')), 'w') as makefile:
        makefile.write(YAV_MAKEFILE)

    logging.info('secret:\n' + secret)

    with open(str(os.path.join(path, 'yav_secret.json')), 'w') as makefile:
        makefile.write(secret)

    changelog_dir = os.path.join(path, 'debian')
    if not os.path.exists(changelog_dir):
        os.mkdir(changelog_dir)
    with open(str(os.path.join(changelog_dir, 'changelog')), 'w') as changelog:
        changelog.write(YAV_CHANGELOG)

    env = os.environ
    env['DONT_SIGN'] = '1'
    run_subprocess('make deb', path, env=env)


class SaneBuildTest(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        _container = sdk2.parameters.Container(
            'Environment container resource',
            default_value=624240993,
            required=True
        )

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            'Arcadia url',
            required=True
        )

        arcadia_patch = sdk2.parameters.String(
            "Apply patch (text diff or rbtorrent)",
            multiline=True
        )

        project_path = sdk2.parameters.String(
            'Project path inside acadia',
            required=True
        )

        with sdk2.parameters.String("Platfrom", required=True) as platform:
            platform.values.trusty = 'trusty'
            platform.values.lucid = 'lucid'
            platform.values.hardy = 'hardy'

        with sdk2.parameters.String("Stage", required=True) as stage:
            stage.values.stable = 'stable'
            stage.values.unstabe = 'unstable'
            stage.values.testing = 'testing'

        ya_make_deps = sdk2.parameters.List(
            "Dependencies packages for 'ya package' build",
            required=False
        )

        yav_secret = sdk2.parameters.Vault(
            "YT secret",
            description="'name' or 'owner:name' for extracting from Vault",
            required=False
        )

    class Requirements(sdk2.Task.Requirements):
        privileged = True

    def run_ya_packages_tasks(self, arcadia_patch, packages):
        subtasks = []
        YaPackageClass = sdk2.Task['YA_PACKAGE']

        descr = 'Created from SaneBuildTest task #' + str(self.id)
        for package in packages:
            task = YaPackageClass(
                self,
                description=descr,
                arcadia_patch=arcadia_patch,
                packages=package,
                publish_package=False
            ).enqueue()
            subtasks.append(task.id)

        return subtasks

    def prepare_build_dir(self):
        arcadia_root = sdk2.vcs.svn.Arcadia.get_arcadia_src_dir(
            self.Parameters.arcadia_url
        )

        sdk2.svn.Arcadia.apply_patch(
            arcadia_root,
            self.Parameters.arcadia_patch,
            str(self.path())
        )

        if os.path.exists(BUILD_DIR):
            shutil.rmtree(BUILD_DIR)

        shutil.copytree(
            str(os.path.join(
                str(arcadia_root),
                str(self.Parameters.project_path)
            )),
            str(os.path.join(
                str(self.path()),
                BUILD_DIR
            ))
        )

        run_subprocess("chmod a+rw -R " + BUILD_DIR, str(self.path()))

    def make_dependencies(self, dst_path):
        if not self.Parameters.ya_make_deps:
            return []

        with self.memoize_stage.create_children:
            self.Context.subtasks = self.run_ya_packages_tasks(
                arcadia_patch=self.Parameters.arcadia_patch,
                packages=self.Parameters.ya_make_deps
            )

            raise sdk2.WaitTask(
                self.Context.subtasks,
                statuses=ctt.Status.Group.FINISH,
                wait_all=True
            )

        subtasks = list(self.find(statuses=ctt.Status.SUCCESS))

        assert len(subtasks) == len(self.Parameters.ya_make_deps)
        assert set(self.Context.subtasks) == set(map(lambda _: _.id, subtasks))

        for task in subtasks:
            res_path = sdk2.ResourceData(
                sdk2.Resource['YA_PACKAGE'].find(task=task).first()
            ).path

            with tarfile.open(str(res_path), 'r:gz') as tar:
                tar.extractall(dst_path)

        return glob.glob(os.path.join(dst_path, '*.deb'))

    def on_prepare(self):
        if not self.Parameters.ya_make_deps or self.Context.subtasks:
            self.prepare_build_dir()

    def on_execute(self):
        build_dir = str(self.path(BUILD_DIR))
        dependencies = self.make_dependencies(build_dir)

        mount_dir(
            dir_path=str(os.path.join(build_dir, os.path.pardir)),
            mount_point=USER_HOME
        )

        if self.Parameters.yav_secret:
            make_yav_secret_deb(YAV_SECRET_DIR, self.Parameters.yav_secret.data())
            dependencies.append(YAV_SECRET_DEB_PATH)

        logging.debug("Deb packages:\n" + str(dependencies))

        run_sane_build_test(
            path=str(os.path.join(USER_HOME, BUILD_DIR)),
            user=USER_NAME,
            platform=self.Parameters.platform,
            stage=self.Parameters.stage,
            dependencies=dependencies
        )

