# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from sandbox import sdk2
from sandbox.common import errors
from sandbox.sandboxsdk import environments
from sandbox.projects.music.MusicSendEmailWithMusicNetworks import MusicSendEmailWithMusicNetworks

import logging
import smtplib


class SendEmailWithAutoNetworks(sdk2.Task):
    """
    Collects Auto ip addr and networks. Sends them as email
    """
    racktables_api = "https://racktables.yandex.net/export"
    nets_api = racktables_api + "/nets-by-project.php"
    ipv4_nets = racktables_api + "/allipv4nets.php"
    hbf_api = "https://hbf.yandex.net/"
    mds_tun64_args = {
        "format": "allipnets",
        "text": "{TUN64 block} and {$vlan_1302}",
    }

    music_domains = (
        "analytics-slb.mobile.yandex.net",
        "api.music.yandex.net",
        "avatars.mds.yandex.net",
        "avatars.yandex.net",
        "match.music.yandex.net",
        "mobileproxy-certificate.passport.yandex.net",
        "music.yandex.ru",
        "social.yandex.ru",
        "storage.mds.yandex.net",
        "storage.music.yandex.net",
        "yapic.yandex.ru",
        "report.appmetrica.yandex.net",
        "startup-slb.mobile.yandex.net",
        "rosenberg.appmetrica.yandex.net",
        "redirect.appmetrica.yandex.com",
        "push.api.appmetrica.yandex.net",
        "proxy-partner.appmetrica.yandex.net",
    )

    weather_domains = (
        "api.weather.yandex.net",
    )

    navi_domains = (
        "mobile.navi.yandex.net",
        "spdy3.mob.maps.yandex.net",
        "vins-navi.voicetech.yandex.net",
        "login.yandex.ru",
        "avatars.yandex.net",
        "oauth.mobile.yandex.net",
        "avatars.mds.yandex.net",
        "banners.mobile.yandex.net",
        "analytics.mobile.yandex.net",
        "settings.crashlytics.com",
        "e.crashlytics.com",
        "m.legal.yandex.com.tr",
        "m.legal.yandex.ru",
        "payment.yandex.net",
    )

    maps_domains = (
        "spdy3.mob.maps.yandex.net",
        "geointernal.mob.maps.yandex.net",
        "mobile.public.voice.yandex.net",
        "mobileproxy.passport.yandex.net",
        "login.yandex.ru",
        "avatars.mds.yandex.net",
        "banners.mobile.yandex.net",
        "analytics.mobile.yandex.net",
        "settings.crashlytics.com",
        "e.crashlytics.com",
        "geointernal.mob.maps.yandex.net",
        "www.yandex.ru",
        "geointernal.mob.maps.yandex.net",
        "static-maps.yandex.ru",
        "mobile.yamobile.yandex.net",
        "storage.mds.yandex.net",
        "http.mob.maps.yandex.net",
        "mobile-maps-common.s3.yandex.net",
        "yandex.ru",
        "proxy.mob.maps.yandex.net",
        "storage.mds.yandex.net",
    )

    updater_domains = (
        "automotive.yandex.net",
        "s3.yandex.net",
        "m.soft.yandex.ru",
    )

    alice_domains = (
        "uniproxy.alice.yandex.net",
    )

    mail_body = "Список IP адресов для обнуления трафика\n\n"
    mail_from = "Yandex Auto <mkrutyakov@yandex-team.ru>"  # FIXME: change to prod email

    mail_subj = "Список ip адресов Яндекс Авто"

    class Parameters(sdk2.Task.Parameters):
        description = "Collects ip addr and networks. Sends them as email"
        sniffed_domains = sdk2.parameters.Resource('Sniffed domains of Ya.Auto head', required=True)
        destination_emails = sdk2.parameters.String(
            "Email addresses for mailer",
            description="Space-separated email addresses list",
            default="mkrutyakov@yandex-team.ru esnovozhilova@yandex-team.ru"
        )

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment("dnspython"),
        ]
        cores = 1
        ram = 1024

    @staticmethod
    def get_addr(hostname, nets):
        """
        Add  addr of a hostname to nets list
        """
        from dns import resolver, rdatatype
        ipvers = [rdatatype.A, rdatatype.AAAA]

        res = resolver.Resolver()
        res.nameservers = ['2a02:6b8::1']  # ns1.yandex.net

        try:
            for ipver in ipvers:
                answer = res.query(hostname, rdtype=ipver)
                for record in answer:
                    nets.add(record.to_text())
        except resolver.NoAnswer as ex:
            logging.info(ex)
        except Exception as ex:
            logging.error(ex)

    def on_execute(self):
        nets = set()
        ips = set()

        MusicSendEmailWithMusicNetworks.get_networks(self.hbf_api, nets)
        MusicSendEmailWithMusicNetworks.get_mds_tun(self.ipv4_nets, self.mds_tun64_args, nets)

        sniffed_domains = []
        sniffed_domains_path = str(sdk2.ResourceData(self.Parameters.sniffed_domains).path)
        with open(sniffed_domains_path) as sniffed_domains_file:
            sniffed_domains = list(map(str.rstrip, sniffed_domains_file))

        domains_lists = [
            self.music_domains,
            self.weather_domains,
            self.navi_domains,
            self.maps_domains,
            self.updater_domains,
            self.alice_domains,
            sniffed_domains,
        ]
        for domains_list in domains_lists:
            for domain in domains_list:
                self.get_addr(domain, ips)

        all = sorted(ips | nets)
        self.mail_body += '\n'.join(all)
        mail_to = self.Parameters.destination_emails.split()
        message = MusicSendEmailWithMusicNetworks.create_mail(mail_to, self.mail_from, self.mail_body, self.mail_subj)
        message["To"] = ','.join(mail_to)
        logging.info(message)
        try:
            server = smtplib.SMTP("yabacks.yandex.ru")
            server.sendmail(self.mail_from, mail_to, message.as_string())
            server.quit()
        except smtplib.SMTPException as exc:
            raise errors.TaskError("Didn't send the mail {}".format(exc))
