# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.common.types.task import Status, Semaphores

from contextlib import closing


class ShootAutoRemoteAccessStress(sdk2.Task):
    """
    Prepare auto remote-access stress environment for shooting.
    Start shooting via tank API.
    """

    def _notifications_only_on_failures(self):
        notifications = []
        for notification in self.Parameters.notifications:
            notifications.append(sdk2.Notification(
                statuses=[Status.FAILURE, Status.EXCEPTION, Status.TIMEOUT],
                recipients=notification.recipients,
                transport=notification.transport))
        return notifications

    class Parameters(sdk2.Task.Parameters):
        description = "Perform necessary setup and shoot at auto remote-access stress"

        nanny_service = sdk2.parameters.String(
            "Nanny service",
            default='maps_auto_remote_access_server_stress',
            required=True
        )
        tank_configs = sdk2.parameters.List(
            "Tank configs",
            default=[
                'maps/automotive/remote_access/server/regression/test_imbalance.cfg',
                'maps/automotive/remote_access/server/regression/test_timings.cfg',
            ],
            required=True
        )
        monitoring_config = sdk2.parameters.String(
            "Monitoring config",
            default='maps/automotive/remote_access/server/regression/telegraf.cfg',
            required=True
        )
        use_public_tanks = sdk2.parameters.Bool(
            "Use public tanks",
            default=False,
            required=True
        )
        tanks = sdk2.parameters.List(
            "List of tanks",
            default=['nanny:maps_core_tanks_load'],
            required=True
        )

        with sdk2.parameters.Group('Emulator database parameters') as emulator_db_block:
            emulator_db_host = sdk2.parameters.String(
                "Emulator db host",
                default='man-5do9ew6u1lu4c7jb.db.yandex.net',
                required=True
            )
            emulator_db_name = sdk2.parameters.String(
                "Emulator db name",
                default='pandora_emulator',
                required=True
            )
            emulator_db_user = sdk2.parameters.String(
                "Emulator db user",
                default='pandora_emulator',
                required=True
            )
            emulator_db_password_secret = sdk2.parameters.YavSecret(
                "Emulator db password secret",
                default='sec-01dvcyjgt31b89mdnrxerhs862',
                required=True
            )
            emulator_db_port = sdk2.parameters.Integer(
                "Emulator db port",
                default=6432,
                required=True
            )

        with sdk2.parameters.Group('Server database parameters') as server_db_block:
            server_db_host = sdk2.parameters.String(
                "Server db host",
                default='man-5do9ew6u1lu4c7jb.db.yandex.net',
                required=True
            )
            server_db_name = sdk2.parameters.String(
                "Server db name",
                default='remote_access_backend_stress',
                required=True
            )
            server_db_user = sdk2.parameters.String(
                "Server db user",
                default='remote_access_backend',
                required=True
            )
            server_db_password_secret = sdk2.parameters.YavSecret(
                "Server db password secret",
                default='sec-01dqz1zbw3gm5z1tejfmxtdtc4',
                required=True
            )
            server_db_port = sdk2.parameters.Integer(
                "Server db port",
                default=6432,
                required=True
            )

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment("psycopg2-binary"),
        ]
        cores = 1
        ram = 1024

    class DatabaseCredentials(object):
        """Credentials to access a database"""

        def __init__(self, db_host, db_name, db_user, db_password, db_port):
            self.host = db_host
            self.name = db_name
            self.user = db_user
            self.password = db_password
            self.port = db_port

        def connect_to_database(self):
            import psycopg2
            return psycopg2.connect(
                host=self.host,
                dbname=self.name,
                user=self.user,
                password=self.password,
                port=self.port,
                target_session_attrs='read-write')

    @staticmethod
    def _setup_emulator(db_credentials):
        with closing(db_credentials.connect_to_database()) as db_connection:
            with db_connection.cursor() as cursor:
                cursor.execute('DELETE FROM car_events;')
                db_connection.commit()

    @staticmethod
    def _setup_server(db_credentials):
        with closing(db_credentials.connect_to_database()) as db_connection:
            with db_connection.cursor() as cursor:
                cursor.execute('UPDATE oauth_tokens SET creation_time = now();')
                db_connection.commit()

    def _perform_shooting(self, nanny_service, config_arc_path, config_monitoring_path, use_public_tanks, tanks):
        description = 'Стрельба через танкапи по {}\n{}'.format(nanny_service, config_arc_path)
        shooting_task = ShootViaTankapi(
            self,
            description=description,
            notifications=self._notifications_only_on_failures(),
            nanny_service=nanny_service,
            config_source='arcadia',
            config_arc_path=config_arc_path,
            use_monitoring=True,
            monitoring_source='arcadia',
            monitoring_arc_path=config_monitoring_path,
            ammo_source='in_config',
            use_public_tanks=use_public_tanks,
            tanks=tanks,
        )
        shooting_task.Requirements.semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(name='MAPS_CORE_TANKS'),
                Semaphores.Acquire(name='MAN_MAPS_CORE_SHOOT_VIA_TANKAPI'),
            ],
            release=(
                Status.Group.BREAK,
                Status.Group.FINISH,
                Status.Group.WAIT,
            ),
        )
        shooting_task.save().enqueue()
        raise sdk2.WaitTask([shooting_task.id], Status.Group.BREAK + Status.Group.FINISH, wait_all=True)

    def on_execute(self):
        emulator_db_credentials = ShootAutoRemoteAccessStress.DatabaseCredentials(
            db_host=self.Parameters.emulator_db_host,
            db_name=self.Parameters.emulator_db_name,
            db_user=self.Parameters.emulator_db_user,
            db_password=self.Parameters.emulator_db_password_secret.data()["password"],
            db_port=self.Parameters.emulator_db_port,
        )
        server_db_credentials = ShootAutoRemoteAccessStress.DatabaseCredentials(
            db_host=self.Parameters.server_db_host,
            db_name=self.Parameters.server_db_name,
            db_user=self.Parameters.server_db_user,
            db_password=self.Parameters.server_db_password_secret.data()["password"],
            db_port=self.Parameters.server_db_port,
        )
        for index, tank_config in enumerate(self.Parameters.tank_configs):
            with self.memoize_stage["shooting_{0}".format(index)]:
                ShootAutoRemoteAccessStress._setup_emulator(emulator_db_credentials)
                ShootAutoRemoteAccessStress._setup_server(server_db_credentials)
                self._perform_shooting(
                    self.Parameters.nanny_service,
                    tank_config,
                    self.Parameters.monitoring_config,
                    self.Parameters.use_public_tanks,
                    self.Parameters.tanks,
                )
