# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from sandbox.projects.maps.MapsBinaryBaseTask import MapsBinaryBaseTask
from sandbox.common.errors import TaskFailure

from sandbox import sdk2


class MapsSwitchDutyPhoneTask(MapsBinaryBaseTask):
    class Requirements(MapsBinaryBaseTask.Requirements):
        pass

    class Parameters(MapsBinaryBaseTask.Parameters):
        # common parameters
        kill_timeout = 600

        with sdk2.parameters.Group('Common settings') as common_block:
            oauth_token = sdk2.parameters.Vault(
                'Oauth token owner and name in vault (separated by colon)',
                required=True
            )
            tvm_id = sdk2.parameters.String(
                'TVM ID to obtain service & user tickets for calendar API',
                required=True
            )
            tvm_secret = sdk2.parameters.Vault(
                'TVM secret owner and name in vault (separated by colon)',
                required=True
            )
            calendar_uid = sdk2.parameters.Vault(
                'Calendar uid owner and name in vault (separated by colon)',
                required=True
            )

        with sdk2.parameters.Group('Phone switcher settings') as phone_block:
            phone = sdk2.parameters.String('Shared phone number', default='1721', required=True)
            use_work_phone = sdk2.parameters.Bool('Switch to work phone', default=False)
            fallback_login = sdk2.parameters.String(
                'Staff login to switch to when calendar is empty', required=True
            )

            abc_service = sdk2.parameters.String(
                'ABC service name', default='vops_maps', required=True
            )
            abc_role = sdk2.parameters.String(
                'ABC service role', default=None, required=False
            )
            cal_id = sdk2.parameters.String(
                'Calendar id (leave empty to get from ABC)',
                default='52168', required=False
            )

        with sdk2.parameters.Group('Notifications settings') as telegram_block:
            use_telegram_notification = sdk2.parameters.Bool(
                'Send notifications to telegram chats', default=True
            )
            with use_telegram_notification.value[True]:
                telegram_chat_id = sdk2.parameters.Integer(
                    'Telegram chat id', default='-1001358152683', required=True
                )
                telegram_bot_token = sdk2.parameters.Vault(
                    'Telegram bot token owner and name in vault (separated by colon)',
                    required=True
                )
                abc_to_chats_mapping = sdk2.parameters.Dict(
                    'Other duty ABC services paired with telegram chat ids', required=False
                )

    @property
    def params(self):
        return self.Parameters

    def on_execute(self):
        env_options = {
            'DUTY_PHONE_SWITCHER_CALENDAR_UID': self.params.calendar_uid.data(),
            'DUTY_PHONE_SWITCHER_TVM_SECRET': self.params.tvm_secret.data(),
            'DUTY_PHONE_SWITCHER_OAUTH_TOKEN': self.params.oauth_token.data()
        }

        option_keys = [
            'tvm_id', 'phone', 'use_work_phone', 'fallback_login', 'abc_service', 'abc_role', 'cal_id'
        ]
        options = self.options_from_parameters(option_keys)
        if self.params.use_telegram_notification:
            options.update({
                'telegram_chat_id': self.params.telegram_chat_id,
                'telegram_bot_token': self.params.telegram_bot_token.data()
            })

        self.run_binary(options=options, env_options=env_options)

        fails = {}
        for abc_service, telegram_chat_id in self.params.abc_to_chats_mapping.iteritems():
            options = {
                'abc_service': abc_service,
                'tvm_id': self.params.tvm_id,
                'telegram_chat_id': telegram_chat_id,
                'telegram_bot_token': self.params.telegram_bot_token.data()
            }
            try:
                self.run_binary(options=options, env_options=env_options)
            except TaskFailure as failure:
                self.set_info('Service {} failure: {}'.format(abc_service, failure))
                fails[abc_service] = failure

        if fails:
            raise TaskFailure('Failed for services: {}'.format(fails.keys()))
