import datetime
import logging
import os
import shlex
import shutil
import tarfile

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, subprocess as sp
import sandbox.projects.common.arcadia.sdk as arcadia_sdk
import sandbox.projects.common.constants as consts
from sandbox.common.errors import TaskFailure
import sandbox.common.types.task as ctt
from sandbox.projects.maps.common.ecstatic_bin import MapsEcstaticToolMixin

BINARY_LOGGER_NAME = 'binary'
LOCAL_RELEASE_DIR = 'release'
LOCAL_RESOURCE_DIR = "resource"
EXTERNAL_JAMS_DATASET_NAME = 'yandex-maps-jams-speeds-external'
VERSION_TIME_FORMAT = '%Y.%m.%d.%H.%M.%S'
TRUNC_SECONDS = 60
PACKAGE_INTERNALS_DIR = os.path.join(LOCAL_RELEASE_DIR, "maps/jams/tools/tie-xml-jams/pkg/maps/jams/tools/tie-xml-jams")

def package_file(file_name):
    return os.path.join(PACKAGE_INTERNALS_DIR, file_name)


class TieXmlJamsRegularYtProcessPackage(sdk2.Resource):
    revision = sdk2.Attributes.Integer("Revision", required=True)
    build_path = sdk2.Attributes.String("Path(from Arcadia root) which should be built", required=True)


class TieXmlJamsRegularYtBuildPackage(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 8
        disk_space = 200 * 1024  # Mb
        ram = 32 * 1024  # Mb

    class Parameters(sdk2.Task.Parameters):
        revision = sdk2.parameters.Integer('Revision', required=True)
        build_path = sdk2.parameters.String('Path(from Arcadia root) which should be built', required=True)

    def on_execute(self):
        resource = TieXmlJamsRegularYtProcessPackage(
            self,
            "binary {} revision {}".format(self.Parameters.build_path, self.Parameters.revision),
            LOCAL_RESOURCE_DIR,
            build_path=self.Parameters.build_path,
            revision=self.Parameters.revision
        )

        resource_data = sdk2.ResourceData(resource)

        with arcadia_sdk.mount_arc_path(sdk2.svn.Arcadia.ARCADIA_TRUNK_URL + '@{}'.format(self.Parameters.revision)) as arcadia:
            arcadia_sdk.do_build(
                consts.YMAKE_BUILD_SYSTEM,
                arcadia,
                [self.Parameters.build_path],
                consts.RELEASE_BUILD_TYPE,
                clear_build=False,
                results_dir=str(resource_data.path)
            )
        resource_data.ready()

TVM_ID_TESTING = 2018380
TVM_SECRET_ID_TESTING = 'sec-01e0fc81jrnfqvfh43fwyawcfd'
TVM_ID_PRODUCTION = 2018378
TVM_SECRET_ID_PRODUCTION = 'sec-01e0fbvg6j85tb2375fqczgbx6'

class TieXmlJamsRegularYtProcess(sdk2.Task, MapsEcstaticToolMixin):
    # Multislot: https://wiki.yandex-team.ru/sandbox/cookbook/#cores1multislot/cores4/cores8
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 7 * 24 * 3600

        with sdk2.parameters.RadioGroup("Environment") as environment:
            environment.values.development = environment.Value(value='development', default=True)
            environment.values.testing = environment.Value(value='testing')
            environment.values.production = environment.Value(value='production')

        build_path = sdk2.parameters.String('Path(from Arcadia root) which should be built', required=True)
        binary_path = sdk2.parameters.String('Path(from Arcadia root) to binary file', required=True)
        binary_resource_id = sdk2.parameters.Integer('Sandbox resource id (will be used instead of build if set)')
        binary_params = sdk2.parameters.String('Parameters to binary', default="")
        revision = sdk2.parameters.Integer('Revision of binary needed', required=True)
        vault_owner = sdk2.parameters.String('Sandbox Vault owner', required=True)
        yt_token_vault_name = sdk2.parameters.String('YT token Sandbox Vault name', required=False, default='YT_TOKEN')
        custom_yt_token = sdk2.parameters.String('Custom token to be used instead of vault derived token', required=False)
        dataset_dir = "ecstatic_dataset"
        graph_version = sdk2.parameters.String('Graph version at YT', default='latest', required=True)
        yt_downloaded_graph_path = "yt_downloaded_graph"
        ban_list = package_file("conf/ban-list.yson")
        binary_config = package_file("conf/tie-upload.json")
        convert_binary = package_file("bin/convert/convert")
        logger_name = "/dev/stderr"

    def now_version(self):
        cur_time = datetime.datetime.utcnow()
        cur_time = cur_time.replace(second=cur_time.second - cur_time.second % TRUNC_SECONDS, microsecond=0)
        return cur_time.strftime(VERSION_TIME_FORMAT)

    def upload_to_ecstatic(self, dataset_dir):
        """
        Upload data in `dataset_dir` to ecstatic with version based on current time
        """
        try:
            # env of ecstatic differ from task and need to be set to "stable" for cookie extraction
            env = self.Parameters.environment if self.Parameters.environment != 'production' else 'stable'
            self.ecstatic(env,
                ['upload',
                EXTERNAL_JAMS_DATASET_NAME + '=' + self.now_version(),
                dataset_dir, '+stable'],
                tvm_id=(TVM_ID_TESTING if self.Parameters.environment != 'production' else TVM_ID_PRODUCTION),
                tvm_secret_id=(TVM_SECRET_ID_TESTING if self.Parameters.environment != 'production' else TVM_SECRET_ID_PRODUCTION)
            )
        except sp.CalledProcessError as e:
            self.set_info(e.output)
            raise TaskFailure('Ecstatic returned ' + str(e.returncode))

    def yt_token(self):
        if self.Parameters.custom_yt_token:
            return self.Parameters.custom_yt_token
        yt_token_vault_name = self.Parameters.yt_token_vault_name
        if yt_token_vault_name:
            return sdk2.Vault.data(self.Parameters.vault_owner, yt_token_vault_name)
        raise TaskFailure("Fail to get YT token")

    def on_prepare(self):
        os.environ["YENV_TYPE"] = self.Parameters.environment
        os.environ["ALZ_API_SVN_REVISION"] = str(self.Parameters.revision)
        os.environ["ALZ_API_YT_TOKEN"] = self.yt_token()
        # Allows one to use resources from a subprocess
        # https://wiki.yandex-team.ru/sandbox/cookbook/#kakpoluchitdannyesinxronizirovatresursizpodprocessazadachi
        os.environ['SYNCHROPHAZOTRON_PATH'] = str(self.synchrophazotron)

    def found_resource(self, build_path, revision):
        return TieXmlJamsRegularYtProcessPackage.find(
            state="READY",
            attrs=dict(build_path=build_path, revision=revision)
        ).first()

    def arcadia_package_resource_path(self, build_path, revision):
        resource = self.found_resource(build_path, revision)
        if resource:
            return str(sdk2.ResourceData(resource).path)

        child = TieXmlJamsRegularYtBuildPackage(
            self,
            description="Build {} revision {} for {}".format(build_path, revision, self.id),
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            revision=revision,
            build_path=build_path,
            vault_owner=self.Parameters.vault_owner,
            yt_token_vault_name=self.Parameters.yt_token_vault_name
        ).enqueue()

        raise sdk2.WaitTask([child.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True,
                            timeout=14400)

        resource = self.found_resource(build_path, revision)
        if resource:
            return str(sdk2.ResourceData(resource).path)
        raise TaskFailure("package resource not found at arcadia")

    def sandbox_package_resource_path(self, resource_id):
        resource = sdk2.Resource.find(id=resource_id).first()
        if resource:
            return str(sdk2.ResourceData(resource).path)
        raise TaskFailure("package resource not found at sandbox")

    def package_dir(self, complete_resource_dir):
        shutil.rmtree(LOCAL_RELEASE_DIR, ignore_errors=True)
        if self.Parameters.binary_resource_id != 0:
            os.makedirs(LOCAL_RELEASE_DIR)
            tar = tarfile.open(complete_resource_dir, "r:gz")
            tar.extractall(LOCAL_RELEASE_DIR)
            tar.close()
        else:
            shutil.copytree(complete_resource_dir, LOCAL_RELEASE_DIR, symlinks=True)
            for root, _, _ in os.walk(LOCAL_RELEASE_DIR):
                os.chmod(root, 0o777)
        return LOCAL_RELEASE_DIR

    def command_args(self, complete_package_dir):
        command = [
            os.path.join(complete_package_dir, self.Parameters.binary_path),
            "--dataset-dir", self.Parameters.dataset_dir,
            "--graph-version", self.Parameters.graph_version,
            "--ban-list", self.Parameters.ban_list,
            "--graph-folder", self.Parameters.yt_downloaded_graph_path,
            "--config", self.Parameters.binary_config,
            "--convert-binary", self.Parameters.convert_binary,
            "--no-upload",
            "--log", self.Parameters.logger_name
        ]
        if self.Parameters.binary_params not in [ None, "" ]:
            command += self.shlex.split(self.Parameters.binary_params)
        return command

    def dataset_dir(self, complete_package_dir):
        with ProcessLog(logger=logging.getLogger(BINARY_LOGGER_NAME)) as pl:
            sp.check_call(self.command_args(complete_package_dir), env=os.environ, stdout=pl.stdout, stderr=pl.stdout)
        return self.Parameters.dataset_dir

    def resource_dir(self):
        if self.Parameters.binary_resource_id is not None and self.Parameters.binary_resource_id != 0:
            return self.sandbox_package_resource_path(self.Parameters.binary_resource_id)
        return self.arcadia_package_resource_path(self.Parameters.build_path, self.Parameters.revision)

    def on_execute(self):
        self.upload_to_ecstatic(
            self.dataset_dir(
                self.package_dir(self.resource_dir())
            )
        )
