import json
import logging
import os
import re
import sandbox.sdk2.ssh as ssh
from sandbox import sdk2
from sandbox.projects.maps.common import utils
from sandbox.sdk2.vcs.svn import Arcadia

from sandbox.common.types.client import Tag


class UpdateMapsBaseImageTrunk(sdk2.Task):

    maps_path = None  # type: str

    class Requirements(sdk2.Requirements):
        client_tags = Tag.GENERIC

    class Parameters(sdk2.Parameters):
        revision = sdk2.parameters.Integer('Arcadia revision for base docker image build', required=True)
        packages_to_update = sdk2.parameters.List('pkg.json files to be updated', required=True)

        with sdk2.parameters.Output:
            commited_revision = sdk2.parameters.Integer('Commited revision', required=True)

    def on_execute(self):
        self.maps_path = Arcadia.get_arcadia_src_dir('svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/maps')
        self.Context.dockerfiles = []
        for pkg_json in self.Parameters.packages_to_update:
            dockerfile = self.update_dockerfile_for(pkg_json, self.Parameters.revision)
            if not dockerfile:
                self.set_info("Couldn't find Dockerfile for {}".format(pkg_json))
            else:
                self.set_info("Update base docker version for: {}".format(dockerfile))
                self.Context.dockerfiles.append(dockerfile)

        commit_message = Arcadia.log('svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia',
                                     revision_from=self.Parameters.revision, limit=1)[0]['msg']
        with ssh.Key(self, key_owner='MAPS_GEOINFRA', key_name='ROBOT_MAPS_SANDBOX_SSH'):
            # Star unpacking isn't supported in PY2 :(
            output, _, _ = utils.commit_to_arcadia(
                self.maps_path,
                'Update base docker image to r{}\n{}'
                .format(self.Parameters.revision, self.sanitize_commit_message(commit_message)),
                user='robot-maps-sandbox',
                with_revprop=['arcanum:check-skip=yes', 'arcanum:review-skip=yes'])
            self.set_info(output)
            match = re.search(r'revision\s+(\d+)', output, re.I)
            if match:
                self.Parameters.commited_revision = int(match.group(1))

    def update_dockerfile_for(self, pkg_json, revision):
        logging.info("update dockerfile for {}".format(pkg_json))
        pkg_json_path = self.local_path(pkg_json)
        with open(pkg_json_path) as f:
            pkg_json_content = json.load(f)
        dockerfile_path = self.find_dockerfile(pkg_json_content)
        if not dockerfile_path:
            return None
        dockerfile_local_path = self.local_path(dockerfile_path)
        self.update_dockerfile_revision(dockerfile_local_path, revision)
        return dockerfile_path

    def local_path(self, filename):
        assert self.maps_path is not None
        assert filename.startswith('maps/')
        return os.path.join(self.maps_path, filename.lstrip('maps/'))

    @staticmethod
    def sanitize_commit_message(commit_message):
        """
        Removes 'REVIEW:12345' marks and lines with notification of skip flags:
        > Note: mandatory review (NEED_REVIEW) was skipped
        > Note: mandatory check (NEED_CHECK) was skipped

        :param commit_message: commit message text
        :return: sanitized commit message
        """
        review_tags_filter = r'[_A-Z]*(REVIEW|SKIP)[_A-Z]*(\s*:\s*\d+)?'
        need_skips_filter = r'^Note: mandatory.*$'
        for re_filter in [review_tags_filter, need_skips_filter]:
            commit_message = re.sub(re_filter, '', commit_message, flags=re.MULTILINE)
        return commit_message.strip()

    @staticmethod
    def find_dockerfile(pkg):
        try:
            for entry in pkg["data"]:
                if entry["destination"]["path"] == "/Dockerfile":
                    return entry["source"]["path"]
        except KeyError as e:
            logging.warn("{}".format(e))
            pass

    @staticmethod
    def update_dockerfile_revision(dockerfile_path, revision):
        import re
        with open(dockerfile_path) as dockerfile:
            data = dockerfile.read()
        logging.info("Contents of {}: {}\n".format(dockerfile_path, data))
        data = re.sub(r'^([ \t]*FROM[ \t]+registry.yandex.net/maps/core-base(?:-\w+)?):\w+',
                      r'\1:{}'.format(revision),
                      data,
                      flags=re.MULTILINE)
        logging.info("Changed contents of {}: {}\n".format(dockerfile_path, data))
        with open(dockerfile_path, "w") as dockerfile:
            dockerfile.write(data)
