from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status

import json
import itertools

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry


VAULT_OWNER = "robot-maps-sandbox"


class UpdateMapsBinaryTask(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space =  1024  # 1GB

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        task_path = sdk2.parameters.String('Path inside arcadia to the directory with ya.make')
        schedulers = sdk2.parameters.String('Comma separated list of schedulers to update',
                                            default='', required=False)
        revision = sdk2.parameters.Integer('Revision from which to build binary or None for trunk',
                                           default=None, required=False)
        make_important = sdk2.parameters.Bool('Make important')
        binary_type = sdk2.parameters.String('Type of binary', default=None, required=False)
        with sdk2.parameters.Output:
            uploaded_binary = sdk2.parameters.Resource("Uploaded binary", required=True)

    def _notifications_only_on_failures(self):
        notifications = []
        for notification in self.Parameters.notifications:
            notifications.append(sdk2.Notification(
                statuses=[Status.FAILURE, Status.EXCEPTION, Status.TIMEOUT],
                recipients=notification.recipients,
                transport=notification.transport))
        return notifications

    @staticmethod
    def _retriable_session(retries=3, delay=1,
                           status_forcelist=itertools.chain(
                               (requests.codes.too_many_requests,), xrange(500, 600))):
        retry = Retry(total=retries, backoff_factor=delay, status_forcelist=status_forcelist, method_whitelist=False)
        adapter = HTTPAdapter(max_retries=retry)
        session = requests.Session()
        session.mount('http://', adapter)
        session.mount('https://', adapter)
        return session

    def on_execute(self):
        with self.memoize_stage['build_binary']:
            description = 'build binary from {path} from {revision}'.format(path=self.Parameters.task_path,
                                                                            revision=self.Parameters.revision)
            arcadia_url = 'arcadia:/arc/trunk/arcadia'
            if self.Parameters.revision:
                arcadia_url += '@{revision}'.format(revision=self.Parameters.revision)
            ya_make_task = sdk2.Task['YA_MAKE'](
                self,
                description=description,
                notifications=self._notifications_only_on_failures(),
                checkout_arcadia_from_url=arcadia_url,
                targets=self.Parameters.task_path,
                use_aapi_fuse=True,
                use_arc_instead_of_aapi=True
            )
            ya_make_task.enqueue()
            self.set_info('Starting YA_MAKE with description: {description}'.format(description=description))
            raise sdk2.WaitTask(ya_make_task.id, ctt.Status.Group.FINISH)

        ya_make_task = self.find(sdk2.Task['YA_MAKE'], status=ctt.Status.Group.SUCCEED).first()
        if not ya_make_task:
            raise TaskFailure('YA_MAKE subtask failed')

        resource = sdk2.Resource['BUILD_OUTPUT'].find(task=ya_make_task).first()
        self.set_info('Binary builded, result is in the resource:{resource_id}'.format(resource_id=resource.id))
        resource.share = True
        if self.Parameters.make_important:
            resource.ttl = 'inf'
        if self.Parameters.revision:
            resource.revision = self.Parameters.revision
        if self.Parameters.binary_type:
            resource.binary_type = self.Parameters.binary_type

        sandbox_token = sdk2.Vault.data(VAULT_OWNER, 'SANDBOX_OAUTH_TOKEN')
        for scheduler in self.Parameters.schedulers.split(','):
            if scheduler:
                scheduler = scheduler.strip()
                response = self._retriable_session().put(
                    'https://sandbox.yandex-team.ru/api/v1.0/scheduler/{scheduler}'.format(scheduler=scheduler),
                    json.dumps({
                        'task': {
                            'custom_fields': [
                                {
                                    'name': 'binary',
                                    'value': resource.id
                                }
                            ]
                        }
                    }),
                    headers={
                        'Content-Type': 'application/json; charset=utf-8',
                        'Authorization': 'Bearer {token}'.format(token=sandbox_token)})
                response.raise_for_status()
                self.set_info('Scheduler:{scheduler} was updated'.format(scheduler=scheduler))

        self.Parameters.uploaded_binary = resource
