from sandbox import sdk2
from sandbox.projects.maps.automotive.store.task import MapsAutoStoreTask
import datetime
import logging
import os
import requests
import textwrap


def mapRowValues(keys, values):
    return {kv[0]: kv[1] for kv in zip(keys, values)}


class MapsAutoStoreCleanupDb(MapsAutoStoreTask):
    """
    Clean-up internal store database
    """

    class Parameters(MapsAutoStoreTask.Parameters):
        dry_run = sdk2.parameters.Bool('Dry run', default=False)

    def oauth(self):
        return {"Authorization": "OAuth " + self.secret["YQL_TOKEN"]}

    def rm_package_rollouts(self, table):
        table.fetch_full_data()
        for row in table.rows:
            kv = mapRowValues(table.column_names, row)
            for k in ["flavor", "type", "mcu", "vendor", "model"]:
                if not kv[k]:
                    kv[k] = ""
            rollout = textwrap.dedent("""
                branch: "%(branch)s"
                headunit {
                    type: "%(type)s"
                    mcu: "%(mcu)s"
                    vendor: "%(vendor)s"
                    model: "%(model)s"
                }
                package_id {
                    name: "%(name)s"
                    flavor: "%(flavor)s"
                    version_code: %(version_code)d
                }""" % kv)
            logging.info("Deleting package rollout %s", kv)
            if self.Parameters.dry_run:
                continue
            rsp = requests.delete(self.store_url + "/store/1.x/rollout/package/config",
                headers=self.oauth(),
                data=rollout)
            if rsp:
                deleted = mapRowValues(table.column_names, row)
                deleted["time_deleted"] = datetime.datetime.utcnow().isoformat()
                yield deleted
            else:
                logging.warning("Cannot delete package rollout %s, status code %s answer %s", kv, rsp.status_code, rsp.text)

    def rm_packages(self, table):
        table.fetch_full_data()
        for row in table.rows:
            kv = mapRowValues(table.column_names, row)
            pkg_id = "{}/{}".format(kv["name"], kv["version_code"])
            if kv["flavor"]:
                pkg_id += "/" + kv["flavor"]
            logging.info("Deleting package %s", pkg_id)
            if self.Parameters.dry_run:
                continue
            rsp = requests.delete(self.store_url + "/store/1.x/package/" + pkg_id,
                headers=self.oauth())
            if rsp:
                kv["time_deleted"] = datetime.datetime.utcnow().isoformat()
                yield kv
            else:
                logging.warning("Cannot delete package %s, status code %s answer %s", pkg_id, rsp.status_code, rsp.text)

    def write_log(self, rows, to_table):
        table = self.yt.TablePath(self.yt_path + "/" + to_table, append=True)
        self.yt.write_table(table, rows, raw=False)

    def on_execute(self):
        from yql.client.parameter_value_builder import YqlParameterValueBuilder
        self.init()

        scriptName = os.path.join(
            os.path.dirname(os.path.abspath(__file__)), "cleanup.yql")
        script = open(scriptName).read()
        query = self.yql.query(script, syntax_version=1)
        params = {
            "$env": YqlParameterValueBuilder.make_string(self.Parameters.environment)
        }
        query.run(parameters=YqlParameterValueBuilder.build_json_map(params))
        results = list(query.get_results(wait=True))
        self.write_log(self.rm_package_rollouts(results[0]), "cleanup_package_rollout")
        self.write_log(self.rm_packages(results[1]), "cleanup_package")
