from sandbox import sdk2
from sandbox.projects.maps.automotive.store.task import MapsAutoStoreTask
import datetime
import logging
import pytz


def packageKey(row):
    return row["app_name"] + "/" + row["md5"]


def firmwareKey(row):
    return row["name"] + "/" + row["version"]


def is_older(s3_object, age):
    return pytz.utc.localize(datetime.datetime.utcnow()) - s3_object.last_modified.astimezone(pytz.utc) > age


class MapsAutoStoreCleanupS3(MapsAutoStoreTask):
    """
    Clean-up internal store S3 buckets
    """

    class Parameters(MapsAutoStoreTask.Parameters):
        dry_run = sdk2.parameters.Bool('Dry run', default=False)

    def init(self):
        super(MapsAutoStoreCleanupS3, self).init()
        import boto3

        if (self.Parameters.environment == "testing"):
            self.main_bucket = "maps-auto-store-internal-testing"
        else:
            self.main_bucket = "maps-auto-store-internal"

        self.s3 = boto3.resource(
            's3',
            endpoint_url='http://s3.mds.yandex.net',
            aws_access_key_id=self.secret["store_s3mds_access_key"],
            aws_secret_access_key=self.secret["store_s3mds_secret_key"])

    def delete(self, bucket, keys):
        if self.Parameters.dry_run:
            # just iterate to trigger generator
            for _ in keys:
                pass
            return
        result = bucket.delete_objects(Delete={"Objects": [{"Key": k} for k in keys]})
        if "Errors" in result:
            for error in result["Errors"]:
                logging.error("S3 error: %s", error)
        elif 'Deleted' in result:
            logging.info("Deleted %d objects", len(result["Deleted"]))

    def old_not_used(self, prefix, table, keyMapper):
        used = set(prefix + keyMapper(row) for row in self.yt.read_table(table))
        for obj in self.s3.Bucket(self.main_bucket).objects.filter(Prefix=prefix):
            if obj.key in used:
                used.remove(obj.key)
            elif is_older(obj, datetime.timedelta(days=7)):
                logging.info("About to delete %s", obj.key)
                yield obj.key
        if used:
            logging.error("Some objects have no S3 files: %s", used)

    def on_execute(self):
        self.init()

        bucket = self.s3.Bucket(self.main_bucket)

        tablePkg = self.yt.TablePath(self.yt_path + "/package", columns=["app_name", "md5"])
        self.delete(bucket, self.old_not_used("package/", tablePkg, packageKey))

        tableFw = self.yt.TablePath(self.yt_path + "/firmware", columns=["name", "version"])
        self.delete(bucket, self.old_not_used("firmware/images/", tableFw, firmwareKey))
