from datetime import date, timedelta
import itertools

from sandbox import sdk2
from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.notification as ctn
import sandbox.projects.resource_types.releasers as releasers


class YaCourierBackendDbBackup(sdk2.Task):
    """
    Archiving old courier positions and route state histories
    """

    class Context(sdk2.Context):
        archiving_tasks_ids = []

    class Parameters(sdk2.Task.Parameters):
        notifications = [
            sdk2.Notification(
                [ctt.Status.FAILURE, ctt.Status.Group.BREAK],
                releasers.b2bgeo_releasers,
                ctn.Transport.TELEGRAM
            )
        ]

        binary = sdk2.parameters.Resource('Resource with archive_old_data binary', required=True)

        days_to_expire = sdk2.parameters.Integer(
            'Number of days after which data is considered to be expired',
            default_value=93
        )

        days_to_expire_non_referenced_data = sdk2.parameters.Integer(
            'Number of days after which non-referenced route state history data are considered to be expired',
            default_value=14
        )

    def execute_archiving_task(self, description, binary, program_args_dict):
        args = list(itertools.chain(*program_args_dict.items()))

        subtask = sdk2.Task["MAPS_BINARY_TASK"](
            self,
            description=description,
            binary=binary,
            binary_name ="archive_old_data",
            args=args,
            vault_env_options={
                "YT_TOKEN": "B2BGEO-DEV:YT_TOKEN",
                "YC_PG_PASSWORD": "B2BGEO-DEV:b2bgeo_ya_courier_backend_db_password"}
        )
        subtask.save().enqueue()

        self.Context.archiving_tasks_ids.append(subtask.id)

        raise sdk2.WaitTask(
            subtask,
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
        )

    def on_execute(self):
        expire_date = date.today() - timedelta(days=self.Parameters.days_to_expire)

        common_program_args = {
            "--environment": "prod",
            "--expire-date": expire_date.isoformat()
        }

        with self.memoize_stage.execute_courier_positions_archiving_task:
            program_args = common_program_args.copy()
            program_args["--profile"] = "courier_position"
            self.execute_archiving_task(
                description="Archiving Ya Courier courier positions older than {}".format(program_args["--expire-date"]),
                binary=self.Parameters.binary,
                program_args_dict=program_args
            )

        with self.memoize_stage.execute_route_state_histories_archiving_task:
            program_args = common_program_args.copy()
            program_args["--profile"] = "route_state_history"
            self.execute_archiving_task(
                description="Archiving Ya Courier route state histories older than {}".format(program_args["--expire-date"]),
                binary=self.Parameters.binary,
                program_args_dict=program_args
            )

        with self.memoize_stage.execute_non_referenced_route_state_histories_archiving_task:
            program_args = common_program_args.copy()
            program_args["--profile"] = "non_referenced_route_state_history"
            program_args["--expire-date"] = (
                    date.today() - timedelta(days=self.Parameters.days_to_expire_non_referenced_data)).isoformat()
            self.execute_archiving_task(
                description="Archiving Ya Courier non-referenced route state histories stored earlier than {}".format(
                    program_args["--expire-date"]),
                binary=self.Parameters.binary,
                program_args_dict=program_args
            )

        for subtask_id in self.Context.archiving_tasks_ids:
            subtask = self.find(id=subtask_id).first()
            if subtask is None:
                raise common.errors.TaskFailure("Error: there is no task with id {}".format(subtask_id))
            elif subtask.status != ctt.Status.SUCCESS:
                raise common.errors.TaskFailure("Some of the child tasks has failed")
