from collections import namedtuple
import base64
import logging
import textwrap
import time
import urlparse

from sandbox import sdk2
from sandbox.projects.maps.b2bgeo.lib.deeplink import YaCourierDeepLinkDecoder
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2 import yav


TrackedCourier = namedtuple('TrackedCourier', [
    'DeviceIDHash',
    'DeviceID',
    'RouteId',
    'CourierId',
    'TrackingBegin',
    'TrackingEnd',
    'DeepLinkTimestamp',
    'GenerationTimestamp',
])


class YaCourierCaptureDeepLink(sdk2.Task):
    """
    Capture couriers build_route_on_map navi deeplinks from Metrika logs
    and save them in a clickhouse table
    """

    class Requirements(sdk2.Requirements):
        environments = [
            PipEnvironment("clickhouse_driver"),
        ]

    class Parameters(sdk2.Task.Parameters):
        events_delay = sdk2.parameters.Integer(
            'Events delay in seconds',
            default_value=600
        )
        check_signature = sdk2.parameters.Bool(
            'Check yacourier parameter signature',
            default_value=False
        )

    def get_route_to_generation_timestamp(self, ch):
        query = textwrap.dedent("""
            SELECT RouteId, toUnixTimestamp(GenerationTimestamp)
            FROM maps_b2bgeo.tracked_couriers
        """)
        return {row[0]: row[1] for row in ch.execute(query)}

    def capture_deeplinks(self, ch):
        route_to_generation_timestamp = self.get_route_to_generation_timestamp(ch)
        query = textwrap.dedent("""
            SELECT
                ParsedParams.Key2[indexOf(ParsedParams.Key1, 'yacourier')] AS yacourier,
                DeviceIDHash, DeviceID, EventTimestamp
            FROM client_events_all
            WHERE
                APIKey = 30488
                AND (EventDate = if((now() - {events_delay}) < today(), today() - 1, today()))
                AND (EventTimestamp > (now() - {events_delay}))
                AND EventName = 'application.open-by-urlscheme'
                AND has(ParsedParams.Key1, 'yacourier')
        """).format(
            events_delay=self.Parameters.events_delay
        )
        if self.Parameters.check_signature:
            key = (yav.Secret('sec-01f1d3h5t5x1sbpqcf97mrbnd1').data()['key']).encode()
        else:
            key = None
        decoder = YaCourierDeepLinkDecoder(key)
        to_delete = set()
        to_insert = {}
        for row in ch.execute(query):
            deviceIDHash = row[1]
            timestamp = row[3]
            logging.info('Found deeplink: EventTimestamp={} DeviceIDHash={}'.format(
                timestamp, deviceIDHash
            ))
            try:
                params = decoder.decode(row[0].encode('ascii'))
            except:
                logging.exception('Error decoding yacourier parameter')
                continue
            if 'version' not in params or params['version'] != '1':
                logging.warn(textwrap.dedent("""
                    Unsupported yacourier params encoding version {},
                    expected version is {}
                """).format(params['version'], 1))
                continue
            courier = TrackedCourier(
                DeviceIDHash=deviceIDHash,
                DeviceID=row[2],
                DeepLinkTimestamp=timestamp,
                RouteId=int(params['route_id']),
                CourierId=int(params['courier_id']),
                TrackingBegin=int(params['tracking_begin']),
                TrackingEnd=int(params['tracking_end']),
                GenerationTimestamp=int(params['timestamp']),
            )
            gen_timestamp = route_to_generation_timestamp.get(courier.RouteId)
            if gen_timestamp is None or gen_timestamp < courier.GenerationTimestamp:
                if gen_timestamp:
                    logging.info('Newer deeplink was opened, old generation timestamp is {}'.format(gen_timestamp))
                    to_delete.add(courier.RouteId)
                logging.info('Add {}'.format(courier))
                route_to_generation_timestamp[courier.RouteId] = courier.GenerationTimestamp
                to_insert[courier.RouteId] = courier
            else:
                logging.info("Ignore the same or older deeplink {}".format(courier))
        return to_delete, to_insert

    def save_tracked_couriers(self, ch, couriers):
        result = ch.execute(
            "INSERT INTO maps_b2bgeo.tracked_couriers({}) VALUES".format(
                ','.join(TrackedCourier._fields)
            ),
            [courier._asdict() for courier in couriers]
        )
        logging.info("Added {} couriers for tracking".format(result))

    def delete_old_deeplinks(self, ch, to_delete):
        if not to_delete:
            return
        # perform deletion synchronously (mutations_sync) before inserting updated deeplinks
        query = textwrap.dedent("""
            ALTER TABLE maps_b2bgeo.tracked_couriers
                DELETE WHERE RouteId IN ({})
                SETTINGS mutations_sync = 2
        """).format(','.join([str(route_id) for route_id in to_delete]))
        ch.execute(query)

    def delete_expired(self, ch):
        ch.execute(
            "ALTER TABLE maps_b2bgeo.tracked_couriers DELETE WHERE TrackingEnd < {}".format(
                int(time.time())
            )
        )

    def on_execute(self):
        from clickhouse_driver import Client
        ch = Client(
            'clickhouse.metrika.yandex.net',
            user='robot-b2bgeo-backend',
            password=yav.Secret('sec-01f0jztzb9m8xfdq9g6655qdp5').data()['password'],
            database='mobgiga'
        )
        to_delete, to_insert = self.capture_deeplinks(ch)
        self.delete_old_deeplinks(ch, to_delete)
        self.save_tracked_couriers(ch, to_insert.values())
        self.delete_expired(ch)
