from collections import namedtuple
import datetime
import logging
import pytz
import requests
import textwrap

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2 import yav


YA_COURIER_URL = "https://courier.yandex.ru/api/v1"


Courier = namedtuple('Courier', [
    'device_id_hash',
    'route_id',
    'courier_id'
])


Position = namedtuple('Position', [
    'latitude',
    'longitude',
    'accuracy',
    'time'
])


class YaCourierCaptureLocation(sdk2.Task):
    """
    Capture couriers location from Metrika logs and post them to the backend
    """

    class Requirements(sdk2.Requirements):
        environments = [
            PipEnvironment("clickhouse_driver"),
        ]

    class Parameters(sdk2.Task.Parameters):
        events_delay = sdk2.parameters.Integer(
            'Events delay in seconds',
            default_value=600
        )

        location_grouping_interval = sdk2.parameters.Integer(
            'Group location events by time interval in seconds',
            default_value=15
        )

    def fetch_positions(self):
        from clickhouse_driver import Client
        ch = Client(
            'clickhouse.metrika.yandex.net',
            user='robot-b2bgeo-backend',
            password=yav.Secret('sec-01f0jztzb9m8xfdq9g6655qdp5').data()['password'],
            database='mobgiga'
        )
        query = textwrap.dedent("""
            SELECT
                DeviceIDHash,
                argMin(
                    (RouteId, CourierId,
                    Latitude, Longitude, LocationPrecision, LocationTimestamp),
                    LocationPrecision) AS Location
            FROM client_events_all
            GLOBAL JOIN maps_b2bgeo.tracked_couriers using DeviceIDHash
            WHERE
                APIKey = 30488
                AND (EventDate = if((now() - {events_delay}) < today(), today() - 1, today()))
                AND (EventTimestamp > (now() - {events_delay}))
                AND (now() BETWEEN TrackingBegin AND TrackingEnd)
                AND (LocationTimestamp > 0)
            GROUP BY
                DeviceIDHash,
                toStartOfInterval(toDateTime(LocationTimestamp), toIntervalSecond({location_grouping_interval})) AS intervalStart
        """).format(
            events_delay=self.Parameters.events_delay,
            location_grouping_interval=self.Parameters.location_grouping_interval
        )
        positionsByCourier = {}
        for row in ch.execute(query):
            position = row[1]
            courier = Courier(
                device_id_hash=row[0],
                route_id=position[0],
                courier_id=position[1]
            )
            positionsByCourier.setdefault(courier, []).append(Position(
                latitude=position[2],
                longitude=position[3],
                accuracy=position[4],
                time=datetime.datetime.fromtimestamp(position[5], tz=pytz.utc).isoformat()
            ))
        return positionsByCourier

    def push_positions(self, positionsByCourier):
        oauthToken = yav.Secret('sec-01ddnqnwyd9y10zw11d59x9dy4').data()['OAuth']
        for courier, positions in positionsByCourier.items():
            url = '{api_url}/couriers/{courier_id}/routes/{route_id}/push-positions'.format(
                api_url=YA_COURIER_URL,
                courier_id=courier.courier_id,
                route_id=courier.route_id
            )
            rsp = requests.post(url,
                params={'source': 'navi-ch'},
                headers={'Authorization': 'OAuth {}'.format(oauthToken)},
                json={'positions': [pos._asdict() for pos in positions]}
            )
            logging.info("{} response status {}: {}".format(
                courier, rsp.status_code, rsp.text))

    def on_execute(self):
        self.push_positions(self.fetch_positions())
