import logging
import json
import requests
from sandbox import sdk2
import sandbox.common.types.resource as ctr


TANKER_URL = 'https://tanker-api.yandex-team.ru'
KEYSETS = [
    {
        'name': 'apikeys_tariff_names',
        'project': 'apikeys',
        'keyset': 'services-config.Tariff:name'
    },
    {
        'name': 'apikeys_tariff_descriptions',
        'project': 'apikeys',
        'keyset': 'services-config.Tariff:description'
    },
    {
        'name': 'apikeys_reasons',
        'project': 'apikeys',
        'keyset': 'services-config.ReasonDict:text'
    },
    {
        'name': 'apikeys_counters',
        'project': 'apikeys',
        'keyset': 'services-config.Unit:name'
    },
    {
        'name': 'courier_push_notifications',
        'project': 'b2bgeo-backend',
        'keyset': 'courier-push-notifications'
    },
    {
        'name': 'common',
        'project': 'b2bgeo-backend',
        'keyset': 'common'
    },
    {
        'name': 'sms_notifications',
        'project': 'b2bgeo-backend',
        'keyset': 'sms-notifications'
    },
    {
        'name': 'status_presettled_comments',
        'project': 'b2bgeo-backend',
        'keyset': 'status-presettled-comments'
    },
    {
        'name': 'solver_errors',
        'project': 'b2bgeo-backend',
        'keyset': 'solver-errors'
    },
    {
        'name': 'test_strings',
        'project': 'b2bgeo-backend',
        'keyset': 'test-strings'
    },
    {
        'name': 'solver_backend',
        'project': 'b2bgeo-backend',
        'keyset': 'solver-backend'
    },
    {
        'name': 'rapidjson',
        'project': 'b2bgeo-backend',
        'keyset': 'rapidjson'
    },
    {
        'name': 'solver_core_errors',
        'project': 'b2bgeo-solver',
        'keyset': 'solver-core-errors'
    },
]
ALLOWED_DIFFERENCE_IN_SIZE = 0.9


class YaCourierI18n(sdk2.Resource):
    """ Translations for Ya Courier Backend """
    pack_tar = 1
    ttl = 'inf'
    revision_date = sdk2.resource.Attributes.String(None)


class YaCourierTankerToSandbox(sdk2.Task):
    """ Task gets translations for Ya Courier Backend from Tanker """

    class Parameters(sdk2.Task.Parameters):
        description = "Get translations for Ya Courier Backend from Tanker"
        max_restarts = 2
        kill_timeout = 5 * 60
        tanker_token = sdk2.parameters.YavSecret('Yav secret with Tanker OAuth token', required=True)

    def on_create(self):
        for keyset in KEYSETS:
            attr = "{}_size".format(keyset['name'])
            setattr(YaCourierI18n, attr, sdk2.resource.Attributes.Integer(0))

    def on_execute(self):
        last_resource = YaCourierI18n.find(
            state=ctr.State.READY
        ).first()
        if last_resource:
            logging.info("Found last resource with id: %d, revision date: %s", last_resource.id,
                         last_resource.revision_date)

        dates = []
        for keyset in KEYSETS:
            try:
                date = _get_keyset_date(keyset, self.Parameters.tanker_token.value())
                dates.append(date)
            except Exception as error:
                raise RuntimeError("Can't get info {}: {}".format(keyset['name'], error))

        max_date = max(dates)
        if last_resource and max_date <= last_resource.revision_date:
            logging.info("Resources are up to date")
            return

        resource = YaCourierI18n(self, "Archive with translations", "i18n")
        resource.revision_date = max_date

        data = sdk2.ResourceData(resource)
        data.path.mkdir(0o755, parents=True, exist_ok=True)

        for keyset in KEYSETS:
            try:
                content = _download_keyset(keyset, self.Parameters.tanker_token.value())
            except Exception as error:
                raise RuntimeError("Can't download {}: {}".format(keyset['name'], error))

            attr = "{}_size".format(keyset['name'])
            try:
                last_size = int(getattr(last_resource, attr, 0))
            except (ValueError, AttributeError):
                last_size = 0
            if last_size and len(content) < ALLOWED_DIFFERENCE_IN_SIZE * last_size:
                raise RuntimeError("Keyset {} length ({}) is significantly less than last length ({})".format(
                                   keyset['name'], len(content), last_size))

            data.path.joinpath("{}.json".format(keyset['name'])).write_bytes(content)
            setattr(resource, attr, len(content))

        data.ready()


def _request_tanker(path, params, token):
    try:
        response = requests.get(
            "{}/{}".format(TANKER_URL, path),
            headers={"Authorization": "OAuth {}".format(token)},
            params=params,
            timeout=30,
            verify=True)
    except Exception as error:
        logging.error("Exception: %s", error)
        raise error

    response.raise_for_status()
    return response.json()


def _get_keyset_date(keyset, token):
    response = _request_tanker(
        "fhistory",
        token=token,
        params={
            'project-id': keyset['project'],
            'keyset-id': keyset['keyset'],
            'limit': '1',
        })

    commits = response.get('commits')
    if not commits:
        return None

    date = commits[0]['date']
    logging.info("Info project=%s, keyset=%s: date=%s", keyset['project'], keyset['keyset'], date)
    return date


def _download_keyset(keyset, token):
    logging.info("Downloading keyset %s", keyset)
    response = _request_tanker(
        "keysets/json/",
        token=token,
        params={
            'project-id': keyset['project'],
            'keyset-id': keyset['keyset'],
            'flat-keyset': '1'
        })

    result = json.dumps(response, indent=4)
    logging.info("Done project=%s, keyset=%s, length: %d", keyset['project'], keyset['keyset'], len(result))
    return result
