import logging
import os
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess

from sandbox.projects.common.arcadia import sdk as arcadia_sdk


ARCADIA_URL = 'arcadia:/arc/trunk/arcadia@{revision}'
PROJECT_PATH = 'maps/b2bgeo/clang-tidy'


def get_ya_clang_tidy_paths():
    clang_tidy_path = subprocess.check_output(
        "./../../../ya tool clang-tidy --print-path", shell=True, universal_newlines=True).strip()
    clang_root_path = os.path.dirname(os.path.dirname(clang_tidy_path))
    return {
        "clang_tidy": clang_tidy_path,
        "run_clang_tidy": os.path.join(clang_root_path, "share/clang/run-clang-tidy.py"),
        "clang_apply_replacements": os.path.join(clang_root_path, "bin/clang-apply-replacements")
    }


class BbgeoClangTidy(sdk2.Task):
    """ clang-tidy task for maps/b2bgeo/clang-tidy"""
    class Parameters(sdk2.Task.Parameters):
        container = sdk2.parameters.Container(
            'Environment container resource',
            required=True,
            default_value='1376326597'
        )
        revision = sdk2.parameters.String(
            'Revision',
            required=True
        )

    def on_execute(self):
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("clang_tidy")) as process_log:
            arcadia_url = ARCADIA_URL.format(revision=self.Parameters.revision)
            with arcadia_sdk.mount_arc_path(arcadia_url) as arcadia_path:
                project_path = os.path.join(arcadia_path, PROJECT_PATH)
                os.chdir(project_path)

                logging.info("Get clang-tidy paths: ")
                paths = get_ya_clang_tidy_paths()
                for key in paths:
                    logging.info("{key}: {path}".format(key=key, path=paths[key]))
                    assert os.path.exists(paths[key])

                logging.info("Print clang-tidy version:")
                subprocess.check_call(paths["clang_tidy"] + " -version", shell=True,
                                      stdout=process_log.stdout, stderr=process_log.stdout)

                # Generate all files needed for clang-tidy build - for example unistat protobuf headers
                # For now only "--add-result .h" is needed
                # Later following options could be required:
                # --add-result .cpp
                # --add-protobuf-result Process protobuf output as a result
                # --add-flatbuf-result Process flatbuf output as a result
                # --replace-result    Build only --add-result targets
                logging.info("Generate files")
                os.chdir("..")
                subprocess.check_call("./../../ya make -r --add-result .h --replace-result",
                                      shell=True, stdout=process_log.stdout, stderr=process_log.stdout)

                logging.info("Create compilation database")
                subprocess.check_call("./../../ya dump compile-commands --no-generated > clang-tidy/compile_commands.json",
                                      shell=True, stdout=process_log.stdout, stderr=process_log.stdout)

                logging.info("Run clang-tidy")
                os.chdir("clang-tidy")
                subprocess.check_call("{run_clang_tidy} -header-filter='.*/maps/b2bgeo/.*(?<!\.pb\.h)$' -clang-tidy-binary {clang_tidy} -clang-apply-replacements-binary {clang_apply_replacements} -fix .*/maps/b2bgeo/.*".format(
                    **paths), shell=True, stdout=process_log.stdout, stderr=process_log.stdout)
