import requests
import logging
import json
from sandbox import sdk2
import sandbox.common.types.misc as ctm
import sandbox.common.types.notification as ctn

from sandbox.projects.maps.common.qloud_client import update_qloud_environment

RELEASE_MSG_INTRO = "Yandex.Routing VRS API is updated. Release notes:"


def _get_image_hash(image_name, token):
    hostname = image_name.split("/")[0]
    image_with_tag = "/".join(image_name.split("/")[1:])
    image = image_with_tag.split(":")[0]
    tag = image_with_tag.split(":")[1]

    assert hostname == 'registry.yandex.net', "Only internal repository is supported"
    assert image, "Image should not be empty"
    assert tag, "Tag should not be emtpy"

    resp = requests.head(
        "https://{host}/v2/{image}/manifests/{tag}".format(
            host=hostname,
            image=image,
            tag=tag),
        headers={
            "Authorization": "OAuth " + token,
            "Accept": "application/vnd.docker.distribution.manifest.v2+json"
        })
    assert resp.ok, "Request {} to docker registry has failed: {} ".format(resp.url, resp.reason)
    return resp.headers["Docker-Content-Digest"]


def _notify_slack(url, changelog):
    requests.post(
        url,
        headers={"Content-Type": "application/json"},
        data=json.dumps({
            "text": RELEASE_MSG_INTRO,
            "attachments": [{
                "text": changelog
            }]
        }))


class BbgeoReleaseSolver(sdk2.Task):
    """
    Updates specified Qloud componnent with specified image
    """
    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Update parameters") as group_1:
            test_deploy_task = sdk2.parameters.Task(
                "Testing deployment task",
                task_type="BBGEO_SOLVER_AUTO_DEPLOY")
            image_name = sdk2.parameters.String(
                "New image name (will be automaticaaly determined from task)")

        with sdk2.parameters.Group("Notification parameters") as notify_group:
            slack_web_hook_vault = sdk2.parameters.String(
                "Vault with Webhook url for Slack notification",
                default_value="b2bgeo_releaser_slack_webhook",
                required=False)

            relese_notification_email = sdk2.parameters.String(
                "Email to send release notification",
                default_value="routing-releases@yandex-team.ru",
                required=False)

        with sdk2.parameters.Group("System parameters") as group_2:
            qloud_environment_id = sdk2.parameters.String(
                "Qloud environment to update",
                default_value="b2bgeo.ya-courier.solver",
                required=True)

            qloud_token = sdk2.parameters.String(
                "Sandbox Vault Name, with OAuth key for Qloud",
                default_value="robot_b2bgeo_qloud_token",
                required=True)

            docker_token = sdk2.parameters.String(
                "Sandbox Vault Name, with OAuth key for docker registry",
                default_value="robot_b2bgeo_docker_token",
                required=True)

            components_list = sdk2.parameters.List(
                "List of components, to update",
                default=["solver-balancer"],
                required=True)

    def on_execute(self):
        parent_task = self.Parameters.test_deploy_task

        if parent_task:
            image_name = parent_task.Context.registry_tags[0]
            logging.info("Image name inherited from task: " + image_name)
            assert not self.Parameters.image_name, "No need for image name if you have specified task"
        else:
            image_name = self.Parameters.image_name
            logging.info("Image name inherited from task: " + image_name)

        image_hash = _get_image_hash(image_name, sdk2.Vault.data(self.Parameters.docker_token))
        changelog = self.Parameters.description
        components = self.Parameters.components_list

        def update_solver_env(env):
            for component in env["components"]:
                if component["componentName"] in components:
                    properties = component["properties"]
                    properties["repository"] = image_name
                    properties["hash"] = image_hash
            env["comment"] = "Image is updated from sandbox\n" + changelog

        logging.info("Updating " + ",".join(components))
        logging.info("Iamge {}, hash: {}".format(image_name, image_hash))
        logging.info("Release notes:\n" + changelog)

        update_qloud_environment(
            self.Parameters.qloud_environment_id,
            sdk2.Vault.data(self.Parameters.qloud_token),
            update_solver_env,
            "bbgeo-bugs@yandex-team.ru")

        if self.Parameters.slack_web_hook_vault:
            _notify_slack(
                sdk2.Vault.data(self.Parameters.slack_web_hook_vault),
                changelog)

        if self.Parameters.relese_notification_email:
            self.server.notification(
                author="Yandex.Routing Team <routing-noreply@yandex-team.ru>",
                subject="Yandex.Routing service release",
                body=RELEASE_MSG_INTRO +"\n" + changelog,
                recipients=[self.Parameters.relese_notification_email, "golovasteek@yandex-team.ru"],
                transport=ctn.Transport.EMAIL)
