import os
import re
import jinja2
import logging
from datetime import datetime
from sandbox import sdk2
from sandbox import common
from sandbox.projects.common.arcadia.sdk import mount_arc_path
from sandbox.sdk2.helpers import subprocess
from sandbox.sandboxsdk.environments import PipEnvironment
from collections import defaultdict

FAILURE_MESSSAGE='''
Ya-Courier Backend integration tests failed. Please fix them ASAP.
'''

ARCADIA_URL = 'arcadia-arc:/#r{svn_revision}'
PROJECT_PATH = 'maps/b2bgeo/ya_courier/backend/bin/tests'
TEST_BACKEND_URL = 'https://test.courier.yandex.ru'
MOBILE_TOKEN_VAULT_TESTING = 'b2bgeo_ya_courier_backend_courier_app_token'
LOGISTICAIN_TOKEN_VAULT_TESTING = 'b2bgeo_ya_courier_backend_logistician_token_testing'
COURIER_TOKEN_VAULT_TESTING = 'b2bgeo_ya_courier_backend_courier_token_testing'
MANAGER_TOKEN_VAULT_TESTING = 'b2bgeo_ya_courier_backend_user_token_testing'
ADMIN_TOKEN_VAULT_TESTING = 'b2bgeo_ya_courier_backend_admin_token_testing'
MANAGER_TOKEN_VAULT_TESTING_2 = 'b2bgeo_ya_courier_backend_user_token_testing_2'
ADMIN_TOKEN_VAULT_TESTING_2 = 'b2bgeo_ya_courier_backend_admin_token_testing_2'
COURIER_SOLVER_APIKEY_TESTING = 'b2bgeo_ya_courier_backend_courier_solver_apikey_testing'
YA_COURIER_TEST_SUPER_USER_TOKEN = 'b2bgeo_ya_courier_backend_super_user_token_testing'
UNREGISTERED_TOKEN_VAULT_TESTING = 'b2bgeo_ya_courier_backend_unregistered_token_testing'
YA_COURIER_TEST_ACCOUNTS = 'b2bgeo_ya_courier_test_accounts'
B2BGEO_STATBOX_OAUTH_TOKEN = 'robot_b2bgeo_statbox_token'


def run_process(args, cwd, env=None):
    process = subprocess.Popen(
        args,
        stdout=subprocess.PIPE,
        stderr=subprocess.STDOUT,
        cwd=cwd,
        env=env)
    out, _ = process.communicate()
    return process.returncode, out


class Reporter(object):

    class TestcaseInfo(object):
        def __init__(self, passed, full_name, duration_sec):
            self.passed = passed
            self.full_name = full_name
            self.duration_sec = duration_sec

        def suite(self):
            parts = self.full_name.split('::')
            return parts[0].replace('.py', '') if parts else "Root"

        def test_name(self):
            return self.full_name.split('::')[-1]

    def __init__(self, testcases):
        self.testcases = testcases

        time_by_suite = defaultdict(float)
        for testcase in self.testcases:
            time_by_suite[testcase.suite()] += testcase.duration_sec
        stats = [{
            'suite': test_suite,
            'duration_sec': test_duration
        } for test_suite, test_duration in time_by_suite.items()]

        self.suite_stats = sorted(stats, key=lambda stat: stat['duration_sec'], reverse=True)
        self.total_duration = sum([test_info.duration_sec for test_info in self.testcases])

    @staticmethod
    def from_tests_output(tests_output):
        pattern = re.compile(r"\[(good|fail)\] (test_.*\.py.*) \[.*\] \((.*) s\)")

        line_matchers = map(pattern.match, [line for line in tests_output.split('\n') if line])

        return Reporter(
            sorted([Reporter.TestcaseInfo(passed=re_match.group(1) == 'good',
                                          full_name=re_match.group(2),
                                          duration_sec=float(re_match.group(3)))
                    for re_match in line_matchers if re_match], key=lambda t: t.duration_sec, reverse=True))

    def html_report(self):
        template_dir = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_dir), extensions=["jinja2.ext.do"])
        return env.get_template("tests_duration_report.html").render(tests=self.testcases,
                                                                     suite_stats=self.suite_stats,
                                                                     total_duration=self.total_duration)


class BbgeoYaCourierIntegrationTests(sdk2.Task):
    '''
    Run Ya-Courier Backend integration tests
    '''
    class Requirements(sdk2.Requirements):
        environments = (
            PipEnvironment('python-statface-client', '0.142.0', use_wheel=True),
        )

    class Parameters(sdk2.Task.Parameters):
        _container = sdk2.parameters.Container(
            'Environment container resource',
            required=True,
            default_value='2318732917'
        )

        revision = sdk2.parameters.String(
            'Svn revision to mount'
        )
        arcadia_url = sdk2.parameters.String(
            'Arcadia url to mount',
            default=''
        )

        kill_timeout = 2 * 60 * 60  # 2 hours

    def on_execute(self):
        from statface_client import StatfaceClientError

        arcadia_url = self.Parameters.arcadia_url \
                      if self.Parameters.arcadia_url \
                      else ARCADIA_URL.format(svn_revision=self.Parameters.revision)
        with mount_arc_path(arcadia_url) as arcadia_path:
            tests_env = os.environ.copy()
            tests_env['YA_COURIER_TESTS_BACKEND_URL'] = TEST_BACKEND_URL
            tests_env['YA_COURIER_TEST_TOKEN'] = sdk2.Vault.data(LOGISTICAIN_TOKEN_VAULT_TESTING)
            tests_env['YA_COURIER_TEST_MOBILE_TOKEN'] = sdk2.Vault.data(MOBILE_TOKEN_VAULT_TESTING)
            tests_env['YA_COURIER_TEST_SUPER_USER_TOKEN'] = sdk2.Vault.data(YA_COURIER_TEST_SUPER_USER_TOKEN)
            tests_env['YA_COURIER_TEST_TOKEN_COURIER'] = sdk2.Vault.data(COURIER_TOKEN_VAULT_TESTING)
            tests_env['YA_COURIER_TEST_TOKEN_ROLE_MANAGER'] = sdk2.Vault.data(MANAGER_TOKEN_VAULT_TESTING)
            tests_env['YA_COURIER_TEST_TOKEN_ROLE_ADMIN'] = sdk2.Vault.data(ADMIN_TOKEN_VAULT_TESTING)
            tests_env['YA_COURIER_TEST_TOKEN_ROLE_MANAGER2'] = sdk2.Vault.data(MANAGER_TOKEN_VAULT_TESTING_2)
            tests_env['YA_COURIER_TEST_TOKEN_ROLE_ADMIN2'] = sdk2.Vault.data(ADMIN_TOKEN_VAULT_TESTING_2)
            tests_env['YA_COURIER_TEST_TOKEN_UNREGISTERED'] = sdk2.Vault.data(UNREGISTERED_TOKEN_VAULT_TESTING)
            tests_env['YA_COURIER_VRP_SOLVER_APIKEY_MVRP'] = sdk2.Vault.data(COURIER_SOLVER_APIKEY_TESTING)
            tests_env['YA_COURIER_VRP_SOLVER_APIKEY_SVRP'] = sdk2.Vault.data(COURIER_SOLVER_APIKEY_TESTING)
            tests_env['YA_COURIER_TEST_ACCOUNTS'] = sdk2.Vault.data(YA_COURIER_TEST_ACCOUNTS)
            statbox_oauth_token = sdk2.Vault.data(B2BGEO_STATBOX_OAUTH_TOKEN)

            project_path = os.path.join(arcadia_path, PROJECT_PATH)
            return_code, out = run_process(['./../../../../../../ya', 'make', '-r', '--test-type', 'pytest', '--run-all-tests', '--show-passed-tests'],
                                           project_path, tests_env)

            reporter = Reporter.from_tests_output(out)
            self.Context.tests_duration_report = reporter.html_report()
            self.Context.save()

            if return_code != 0:
                raise common.errors.TaskFailure(FAILURE_MESSSAGE + out)


    @sdk2.report(title="Test running times")
    def tests_duration(self):
        return self.Context.tests_duration_report or "Report is not ready yet. Try again later."
