import logging
import requests
from requests.exceptions import RequestException
import xml.etree.ElementTree as ElementTree

from sandbox import sdk2, common
from sandbox.projects.maps.common.retry import retry
from sandbox.sandboxsdk.environments import PipEnvironment

TEAMCITY_API_URL = "https://teamcity.yandex-team.ru"


class NotFinishedException(Exception):
    pass


class BbgeoDocumentationDeploy(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        build_task_id = sdk2.parameters.String("Id of documentation deploy task in Teamcity", default="DocContent_Routing_BuildStableAndDeployToTesting", required=True)

        build_delivery = sdk2.parameters.Bool("Update delivery swagger", default=False, required=True)
        build_vrs = sdk2.parameters.Bool("Update VRP swagger", default=False, required=True)
        svn_rev = sdk2.parameters.String("Docs arc version")
        svn_rev_delivery = sdk2.parameters.String("Delivery swagger version")
        svn_rev_vrp = sdk2.parameters.String("VRP swagger version")
        timeout = sdk2.parameters.Integer("Timeout to request Teamcity build result in seconds", default=20 * 60)


    def _get_teamcity_result_with_retries(self, task_href, headers):
        @retry(
            exceptions=(NotFinishedException, RequestException),
            tries=self.Parameters.timeout // 5 + 1,
            delay=5)
        def _get_teamcity_result(task_href, headers):
            response = requests.get(TEAMCITY_API_URL + task_href, headers=headers)
            response.raise_for_status()
            root = ElementTree.fromstring(response.text)
            task_state = root.attrib["state"]
            finished = (task_state == "finished")
            if finished:
                task_status = root.attrib["status"]
                if task_status != "SUCCESS":
                    raise common.errors.TaskFailure("Teamcity task finished with status {}".format(task_status))
                return
            raise NotFinishedException

        _get_teamcity_result(task_href, headers)

    def on_execute(self):
        teamcity_oauth_token = sdk2.Vault.data('robot_b2bgeo_teamcity_token')
        headers = {
            "Authorization": "OAuth {}".format(teamcity_oauth_token),
            "Content-Type": "application/xml",
        }

        build = ElementTree.Element("build")
        ElementTree.SubElement(build, "buildType", id=self.Parameters.build_task_id)
        properties = ElementTree.SubElement(build, "properties")

        ElementTree.SubElement(properties, "property", name="build_delivery", value=str(self.Parameters.build_delivery))
        ElementTree.SubElement(properties, "property", name="build_vrs", value=str(self.Parameters.build_vrs))

        if self.Parameters.svn_rev:
            ElementTree.SubElement(properties, "property", name="svn_rev", value=self.Parameters.svn_rev)
        if self.Parameters.svn_rev_delivery:
            ElementTree.SubElement(properties, "property", name="svn_rev_delivery", value=self.Parameters.svn_rev_delivery)
        if self.Parameters.svn_rev_vrp:
            ElementTree.SubElement(properties, "property", name="svn_rev_vrp", value=self.Parameters.svn_rev_vrp)

        request_data = ElementTree.tostring(build, encoding="utf-8")
        logging.info("Request data: %s", request_data)
        response = requests.post(TEAMCITY_API_URL + "/app/rest/buildQueue", headers=headers, data=request_data).text
        logging.info("Server response: %s", response)

        root = ElementTree.fromstring(response)
        task_url = root.attrib.get("webUrl")
        self.set_info("Teamcity URL: {}".format(task_url))

        self._get_teamcity_result_with_retries(root.attrib["href"], headers)
