import os
import json

from sandbox import sdk2
import sandbox.projects.maps.carparks.common.utils as utils
from sandbox.projects.maps.carparks.common.task import MapsCarparksBaseTask


def get_binary(resource):
    return str(sdk2.ResourceData(resource).path)


class MapsCarparksMinerTask(MapsCarparksBaseTask):
    class Parameters(MapsCarparksBaseTask.Parameters):
        kill_timeout = 7 * 24 * 3600

        yql_token_vault_name = sdk2.parameters.String(
            'YQL token Sandbox Vault name',
            required=True,
            default='YQL_TOKEN'
        )

        sprav_company_export_path = sdk2.parameters.String(
            'Sprav company export',
            default='//home/sprav/altay/prod/snapshot/company'
        )

        yt_mined_snippets_dir = sdk2.parameters.String(
            'Mined snippets directory',
            required=True,
            description='for example:'
                        '//home/maps/carparks/testing/daps/snippets/mined'
        )

        miner_config_arc_path = sdk2.parameters.String(
            "Path to carparks miner config",
            required=True,
            description='for example: maps/carparks/tools/'
                        'carparks_miner/config/carparks_miner.testing.json'
        )

        dap_snippets_config_arc_path = sdk2.parameters.String(
            "Path to dap snippets config",
            required=True,
            description='for example: maps/carparks/tools/'
                        'dap_snippets/config/config.testing.json'
        )
        
        with sdk2.parameters.Group('Binaries') as binaries_group:
            carparks_miner_binary = sdk2.parameters.Resource(
                'Carparks miner binary'
            )
            dap_snippets_binary = sdk2.parameters.Resource(
                'DAP snippets binary'
            )

    def _get_file_from_arcadia(self, arcadia_path, local_path):
        arcadia_url = sdk2.svn.Arcadia.trunk_url(arcadia_path)

        sdk2.svn.Arcadia.export(arcadia_url, local_path)

    def _carpaks_miner_local_config_path(self):
        return os.getcwd() + '/miner_config.json'

    def _dap_snippets_local_config_path(self):
        return os.getcwd() + '/snippets_config.json'

    def _download_miner_config(self):
        self._get_file_from_arcadia(
            self.Parameters.miner_config_arc_path,
            self._carpaks_miner_local_config_path()
        )

    def _download_snippets_config(self):
        self._get_file_from_arcadia(
            self.Parameters.dap_snippets_config_arc_path,
            self._dap_snippets_local_config_path()
        )

    def _prepare_yql_token(self):
        os.environ["YQL_TOKEN"] = sdk2.Vault.data(
            self.Parameters.vault_owner,
            self.Parameters.yql_token_vault_name)

    def _get_latest_mined_table(self):
        self._download_miner_config()

        with open(self._carpaks_miner_local_config_path()) as config_file:
            config = json.load(config_file)
            mined_carparks_dir = config.get("yt_root")

        path = self.get_latest_node(mined_carparks_dir)
        if not path:
            raise RuntimeError(
                'No mined table found in {}'.format(mined_carparks_dir))

        day = os.path.basename(path)
        table = path + '/clusterized_points'

        if not self.yt.exists(table):
            raise RuntimeError('Table {} does not exist'.format(table))

        return day, table

    def _get_carparks_dataset(self):
        path = self.get_specific_node_or_latest(
            self.Parameters.yt_carparks_dataset_path)
        if not path or not self.yt.exists(path):
            raise RuntimeError('No carparks dataset found')

        return path

    def _run_carparks_miner(self):
        self._download_miner_config()

        args = [self._carpaks_miner_local_config_path()]

        utils.run(self.Parameters.carparks_miner_binary, *args)

    def _make_snippets(self):
        day, mined_table = self._get_latest_mined_table()
        output_dir = self.Parameters.yt_mined_snippets_dir + '/' + day

        self._download_snippets_config()
        config_args = ['--config', self._dap_snippets_local_config_path()]

        utils.run(self.Parameters.dap_snippets_binary,
                  '--generate', mined_table,
                  '--output', output_dir,
                  '--sprav', self.Parameters.sprav_company_export_path,
                  *config_args)

        all_filtered_table = output_dir + '/all_filtered'
        utils.run(self.Parameters.dap_snippets_binary,
                  '--filter-errors', output_dir + '/all',
                  '--output', all_filtered_table,
                  *config_args)

        # Make organizations table to be used as a source of Auto-points
        # in Sprav
        utils.run(self.Parameters.dap_snippets_binary,
                  '--split', all_filtered_table,
                  '--output', output_dir,
                  *config_args)

        return output_dir

    def on_execute(self):
        self.prepare_yt_token()
        self._prepare_yql_token()

        if self.Parameters.carparks_miner_binary:
            self._run_carparks_miner()

        if self.Parameters.dap_snippets_binary:
            output_dir = self._make_snippets()
            self.yt.link(
                output_dir, self.Parameters.yt_mined_snippets_dir + '/current',
                force=True
            )
