import datetime
import logging
import os

from sandbox import sdk2
import sandbox.projects.maps.carparks.common.utils as utils
from sandbox.projects.maps.carparks.common.task import MapsCarparksBaseTask


class MapsCarparksSnippetsTask(MapsCarparksBaseTask):
    class Parameters(MapsCarparksBaseTask.Parameters):
        kill_timeout = 3 * 3600

        dap_snippets_binary = sdk2.parameters.Resource(
            'DAP snippets binary',
            required=True
        )

        nyak_export_path = sdk2.parameters.String(
            'Nyak export (set //<path>/LATEST for the latest table)',
            default='//home/maps/core/garden/stable/carparks_daps/nyak_export/LATEST'
        )

        sprav_company_export_path = sdk2.parameters.String(
            'Sprav company export',
            default='//home/sprav/altay/prod/snapshot/company'
        )

        yt_snippets_root_dir = sdk2.parameters.String(
            'Snippets root directory',
            required=True,
            description='for example:'
                        '//home/maps/carparks/testing/daps/snippets'
        )

        config_arc_path = sdk2.parameters.String(
            "Path to config",
            required=False,
            description='for example: maps/carparks/tools/'
                        'dap_snippets/config/config.testing.json'
        )

    @property
    def _yt_nyak_export_table(self):
        if not self.Parameters.nyak_export_path:
            return None

        return self.get_specific_node_or_latest(self.Parameters.nyak_export_path)

    def _local_config_path(self):
        return os.getcwd() + '/config.json'

    def _download_config(self):
        arcadia_url = sdk2.svn.Arcadia.trunk_url(
            self.Parameters.config_arc_path
        )

        sdk2.svn.Arcadia.export(arcadia_url, self._local_config_path())

    def _create_output_dir_if_not_exists(self):
        date_str = datetime.date.today().strftime("%Y-%m-%d")
        path = self.Parameters.yt_snippets_root_dir + '/merged/' + date_str
        if not self.yt.exists(path):
            self.yt.create('map_node', path)

        return path

    def _merge_snippets(self):
        output_dir = self._create_output_dir_if_not_exists()
        merged_table = output_dir + '/all_merged'

        merge_command = [
            self.Parameters.dap_snippets_binary,
            '--merge', self.Parameters.yt_snippets_root_dir +
            '/mined/current/all_filtered',
            '--output', merged_table]

        if self.Parameters.config_arc_path is not None:
            self._download_config()
            merge_command.extend([
                '--config', self._local_config_path()])
            
        nyak_export_table = self._yt_nyak_export_table
        if nyak_export_table:
            merge_command.extend(['--nyak', nyak_export_table])
        else:
            logging.warn('No nyak export table found')

        if self.Parameters.sprav_company_export_path:
            merge_command.extend([
                '--sprav', self.Parameters.sprav_company_export_path])

        utils.run(*merge_command)

        utils.run(self.Parameters.dap_snippets_binary,
                  '--split', merged_table,
                  '--output', output_dir)

        return output_dir

    def on_execute(self):
        self.prepare_yt_token()

        output_dir = self._merge_snippets()

        self.yt.link(
            output_dir, self.Parameters.yt_snippets_root_dir +
            '/merged/current', force=True
        )
