import os
import logging

import sandbox.common.types.client as ctc
import sandbox.sandboxsdk.parameters as parameters
import sandbox.sandboxsdk.environments as environments
import sandbox.sandboxsdk.process as process

import sandbox.projects.common.build.ArcadiaTask as arcadia_task
from sandbox.projects.common import utils
import sandbox.projects.common.arcadia.sdk as arcadia_sdk
import sandbox.projects.common.build.parameters as common_params


class ProgramParam(parameters.SandboxStringParameter):
    name = 'program'
    description = 'Program to build and run'
    default_value = 'maps/analyzer/check_graph_topology/rips/find_rips'


class ArgsParam(parameters.SandboxStringParameter):
    name = 'program_args'
    description = 'Args to program'
    default_value = ''


class PrepareRipsHypotheses(arcadia_task.ArcadiaTask):
    type = "PREPARE_RIPS_HYPOTHESES"
    description = "Prepares and stores rips hypotheses"
    environment = [environments.SvnEnvironment()]
    client_tags = ctc.Tag.GENERIC
    cores = 17
    input_parameters = [
        parameters.SandboxArcadiaUrlParameter,
        ProgramParam,
        ArgsParam,
        common_params.EnvironmentVarsParam
    ]

    def __init__(self, *args, **kwargs):
        arcadia_task.ArcadiaTask.__init__(self, *args, **kwargs)

    def on_execute(self):
        program_path = utils.get_or_default(self.ctx, ProgramParam)
        env_vars = self.get_env_vars()

        build_target = os.path.dirname(program_path)

        arcadia_dir = self.get_arcadia_src_dir()
        logging.debug("Arcadia src dir %s", arcadia_dir)

        logging.info("Target to build %s", build_target)
        build_return_code = arcadia_sdk.do_build('ya', arcadia_dir, [build_target], clear_build=False, results_dir=arcadia_dir)

        logging.info("Build returned %s", build_return_code)

        process.run_process(
            [os.path.join(arcadia_dir, program_path)] + utils.get_or_default(self.ctx, ArgsParam).split(),
            log_prefix='program', outputs_to_one_file=False, shell=True, work_dir=arcadia_dir, environment=env_vars
        )


__Task__ = PrepareRipsHypotheses
