from sandbox import sdk2
from sandbox.common.types.resource import State as ResourceState
from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.maps.common.base_resources import (
    MapsEcstaticToolBinaryResource,
    MapsEcstaticDatasetResource
)
from sandbox.projects.common import network

import re
import os
import logging

ECSTATIC_ENVIRONMENTS = ['unstable', 'testing', 'datatesting', 'stable']


class MapsEcstaticToolMixin:
    """Mixin to work with ecstatic tool binary"""
    DEFAULT_BINARY_RESOURCE_ID = 1864115444
    debug_ymtorrent = False

    @staticmethod
    def _current_task_name():
        def to_underscope(s):
            # CamelCase class naming to CAMEL_CASE
            # Stack-overflow-driven development
            # https://stackoverflow.com/questions/1175208/elegant-python-function-to-convert-camelcase-to-snake-case
            s1 = re.sub('(.)([A-Z][a-z]+)', r'\1_\2', s)
            return re.sub('([a-z0-9])([A-Z])', r'\1_\2', s1).upper()
        return to_underscope(sdk2.Task.current.type.__name__)

    @staticmethod
    def _ecstatic_host(task_name):
        return 'SANDBOX_' + task_name

    @staticmethod
    def _ecstatic_tool(resource_id):
        ecstatic_resource = MapsEcstaticToolBinaryResource.find(id=resource_id).first()
        logging.info('Using binary from ' + str(ecstatic_resource))
        resource_data = sdk2.ResourceData(ecstatic_resource)
        ecstatic_bin = str(resource_data.path.joinpath('ecstatic'))
        return ecstatic_bin

    @classmethod
    def ecstatic(cls, environment, cmd_args,
                 resource_id=DEFAULT_BINARY_RESOURCE_ID,
                 do_auth=True, extra_env_vars=None,
                 libtorrent_options=None, tvm_id=None, tvm_secret_id=None):
        """
        Args:
            libtorrent_options dict (str->str); use 'true' / 'false' for flag options
        """
        ecstatic_bin = cls._ecstatic_tool(resource_id)

        cmdline = [ecstatic_bin] + cmd_args
        if cls.debug_ymtorrent:
            cmdline.append('--debug-torrent')
        if libtorrent_options:
            for k, v in libtorrent_options.items():
                cmdline.extend(['--libtorrent-option', k, v])

        env_vars = os.environ.copy()
        env_vars['ECSTATIC_IN_SANDBOX'] = 'yes'
        env_vars['ENVIRONMENT_NAME'] = environment

        if do_auth:
            task_name = cls._current_task_name()
            env_vars['ECSTATIC_HOST'] = cls._ecstatic_host(task_name)
            # To use tvm auth you must delegate yav secret to use in sandbox
            # More info: https://wiki.yandex-team.ru/sandbox/yav/
            env_vars['ECSTATIC_TVM_ID'] = str(tvm_id)
            secret_data = sdk2.yav.Secret(tvm_secret_id).data()
            if 'client_secret' in secret_data.keys():
                env_vars['ECSTATIC_TVM_SECRET'] = secret_data['client_secret']
            else:
                assert len(secret_data.keys()) == 1, 'Cannot deduce secret field'
                env_vars['ECSTATIC_TVM_SECRET'] = secret_data[secret_data.keys()[0]]

        if extra_env_vars:
            env_vars.update(extra_env_vars)

        addr = network.get_my_ipv6()
        logging.info('Ecstatic is run inside container with {} ip'.format(addr))

        with sdk2.helpers.ProcessRegistry:
            logging.debug('Run: {}'.format(cmdline))
            return cls._run_command(cmdline, env_vars)

    @staticmethod
    def _run_command(cmdline, env_vars):
        with sdk2.helpers.ProcessLog(logger=logging.getLogger('ecstatic_tool')) as pl:
            output = sdk2.helpers.subprocess.check_output(cmdline, env=env_vars, stderr=pl.stdout)
            logging.info('ecstatic tool output: ' + output)
            return output


class ReleaseMapsEcstaticDatasetResource(MapsEcstaticToolMixin):
    """Mixin class for releasing ecstatic datasets from sandbox resource, overrides method `on_release`."""

    # Override to specify branch for every environment: stable, testing, development
    def environment_branches(self, environment):
        raise RuntimeError('Ecstatic branch for {} unspecified'.format(environment))

    def releaseable_resources(self):
        resources = [resource for resource in sdk2.Resource.find(
            resource_type=MapsEcstaticDatasetResource,
            task=self,
            state=ResourceState.READY
        ).limit(0) if resource.type.releasable]
        return resources

    def ecstatic_branches(self, environment):
        branches = self.environment_branches(environment)
        if isinstance(branches, str):
            branches = [branches]
        for branch in branches:
            if branch[:1] in '+-':
                raise RuntimeError('Should specify branch name without "+", got {}'.format(branch))
        return branches

    def ecstatic_upload(self, environment, dataset, version, directory, branches):
        dataset = '{}={}'.format(dataset, version)
        branches = ['+' + branch for branch in branches]
        logging.info('Uploading {} to {} ecstatic'.format(dataset, environment))
        self.ecstatic(environment, 'upload', dataset, directory, *branches)
        logging.info('Uploaded {}, moved to {}'.format(dataset, ', '.join(branches)))

    def release_resource(self, environment, branches, resource):
        resource_data = sdk2.ResourceData(resource)
        self.ecstatic_upload(
            environment,
            resource.dataset,
            resource.version,
            str(resource_data.path),
            branches
        )

    @property
    def release_template(self):
        return sdk2.ReleaseTemplate(
            types=[
                ReleaseStatus.STABLE,
                ReleaseStatus.TESTING,
                ReleaseStatus.UNSTABLE
            ]
        )

    def on_release(self, parameters):
        environment = parameters['release_status']
        branches = self.ecstatic_branches(environment)
        for resource in self.releaseable_resources():
            self.release_resource(environment, branches, resource)
        self.mark_released_resources(parameters['release_status'])
