import os
import shutil

import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.projects.maps.common.ecstatic_bin import MapsEcstaticToolMixin
from sandbox.sdk2.helpers import subprocess
from sandbox.sandboxsdk import environments

from sandbox.projects.maps.MapsBinaryResourceBuilder import MapsBinaryResource, MapsBinaryResourceBuilder

BINARY_LOCAL_RELEASE_DIR = 'release'


class JamsBuildTaskCommon(MapsEcstaticToolMixin, sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        disk_space = 100 * 1024
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('retry'),
        )

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 8 * 60 * 60

        with sdk2.parameters.RadioGroup('Environment') as environment:
            environment.values.development = environment.Value(value='development', default=True)
            environment.values.testing = environment.Value(value='testing')
            environment.values.production = environment.Value(value='production')

        vault_owner = sdk2.parameters.String('Sandbox Vault owner', required=True)
        yt_token_vault_name = sdk2.parameters.String('YT token Sandbox Vault name', required=True, default='YT_TOKEN')

        binary_revision = sdk2.parameters.Integer('Revision', required=True)
        binary_package_path = sdk2.parameters.String('Path (from Arcadia root) to the package', required=True)

    def on_prepare(self):
        os.environ['YT_TOKEN'] = sdk2.Vault.data(self.Parameters.vault_owner, self.Parameters.yt_token_vault_name)
        os.environ['YT_PROXY'] = 'hahn.yt.yandex.net'
        os.environ['YT_LOG_LEVEL'] = 'INFO'

    def binary_resource(self):
        resource = MapsBinaryResource.find(
            state='READY',
            attrs=dict(
                build_path=self.Parameters.binary_package_path,
                revision=self.Parameters.binary_revision)
        ).first()

        if resource:
            resource_dir = str(sdk2.ResourceData(resource).path)
            shutil.rmtree(BINARY_LOCAL_RELEASE_DIR, ignore_errors=True)
            shutil.copytree(resource_dir, BINARY_LOCAL_RELEASE_DIR, symlinks=True)
            for root, _, _ in os.walk(BINARY_LOCAL_RELEASE_DIR):
                os.chmod(root, 0o777)
            return BINARY_LOCAL_RELEASE_DIR

        child = MapsBinaryResourceBuilder(
            self,
            description='Build {} revision {} for {}'.format(
                self.Parameters.binary_package_path, self.Parameters.binary_revision, self.id),
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            revision=self.Parameters.binary_revision,
            build_path=self.Parameters.binary_package_path,
        ).enqueue()

        raise sdk2.WaitTask([child.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True,
                            timeout=14400)

    def ecstatic_upload(self, path, dataset, version, tag=None):
        dataset_spec = '{}:{}={}'.format(dataset, tag, version) if tag else '{}={}'.format(dataset, version)

        self.wrapped_ecstatic(
            ['upload', dataset_spec, path, '+stable'])

        self.set_info('Uploaded: {}'.format(dataset_spec))

    def ecstatic_versions(self, dataset):
        versions = self.wrapped_ecstatic(['versions', dataset])
        return versions.split()

    def prepare_ecstatic(self, tvm_ids, tvm_sec_uuids):
        from retry import retry

        def wrapped_ecstatic_call(args_list):
            try:
                # ecstatic uses stable instead of production
                env = self.Parameters.environment if self.Parameters.environment != 'production' else 'stable'

                return retry(
                    exceptions=subprocess.CalledProcessError,
                    tries=5,
                    delay=30,
                    max_delay=120,
                    backoff=2)(lambda *args: self.ecstatic(
                        *args,
                        tvm_id=tvm_ids.get(env),
                        tvm_secret_id=tvm_sec_uuids.get(env)))(env, args_list)

            except subprocess.CalledProcessError as e:
                message = 'Ecstatic error code: {}, message: {}'.format(e.returncode, e.output)
                self.set_info(message)
                raise e

        self.wrapped_ecstatic = wrapped_ecstatic_call
