from sandbox import sdk2
from sandbox.projects.common.juggler import jclient


class TaskJugglerReportWithParameters(sdk2.Task):
    """
    Base class that adds Juggler reports on success/failure status

    By default adds:
        1. Checkbox that enables Juggler reports
        2. Host and service name fields

    Check out our tutorial on "How to enable Juggler reports from Sandbox Task"
    https://wiki.yandex-team.ru/geo/quality/documentation/juggler-status-from-sandbox-tasks/
    """

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Juggler parameters') as juggler_parameters:
            juggler_report_is_enabled = sdk2.parameters.Bool(
                'Enable Juggler reports', default=False)
            juggler_host_name = sdk2.parameters.String(
                'Host name', required=False)
            juggler_service_name = sdk2.parameters.String(
                'Service name', required=False)

    @property
    def juggler_report_is_enabled(self):
        return self.Parameters.juggler_report_is_enabled

    @property
    def juggler_host_name(self):
        return self.Parameters.juggler_host_name

    @property
    def juggler_service_name(self):
        return self.Parameters.juggler_service_name

    @property
    def juggler_sandbox_url(self):
        return 'https://sandbox.yandex-team.ru/task/{}'.format(self.id)

    @property
    def juggler_author_staff_url(self):
        return 'https://staff.yandex-team.ru/{}'.format(self.author)

    @property
    def juggler_task_creation_date(self):
        return self.created.strftime('%Y-%m-%d %H:%M:%S')

    @property
    def juggler_task_description(self):
        return (
            '{sandbox_type} {sandbox_url} (by {staff_url}) '
            'created at {create_date} is in {sandbox_status} status'.format(
                sandbox_type=self.type,
                sandbox_url=self.juggler_sandbox_url,
                staff_url=self.juggler_author_staff_url,
                create_date=self.juggler_task_creation_date,
                sandbox_status=self.status))

    def on_break(self, prev_status, status):
        description = (
            '{task_description}: '
            'previous {previous_status} -> current {current_status}'.format(
                task_description=self.juggler_task_description,
                previous_status=prev_status, current_status=status))
        self.send_status_to_juggler(
            status='CRIT',
            description=description)
        super(TaskJugglerReportWithParameters, self).on_break(
            prev_status, status)

    def on_success(self, prev_status):
        self.send_status_to_juggler(status='OK')
        super(TaskJugglerReportWithParameters, self).on_success(prev_status)

    def on_failure(self, prev_status):
        description = (
            '{task_description}: '
            'previous {previous_status} -> current FAILURE'.format(
                task_description=self.juggler_task_description,
                previous_status=prev_status))
        self.send_status_to_juggler(
            status='CRIT',
            description=description)
        super(TaskJugglerReportWithParameters, self).on_failure(prev_status)

    def send_status_to_juggler(
        self, status, host=None, service=None, description=None
    ):
        if self.juggler_report_is_enabled:
            jclient.send_events_to_juggler(
                host=(
                    host
                    if host is not None
                    else self.juggler_host_name),
                service=(
                    service
                    if service is not None
                    else self.juggler_service_name),
                status=status,
                description=(
                    description
                    if description is not None
                    else self.juggler_task_description))
