from sandbox import sdk2


def get_last_resource(resource_type, **kwargs):
    """Finds last uploaded resource.

    resource_type : sdk2.Resource

    **kwargs : `find` arguments

    ngng@: it's very hard to determine what search arguments are suitable
        for `find` query, but here are some of them

    id    : Resource ID
    task  : A task, that created desired resource
    state : Resource state {'READY', 'NOT_READY', 'BROKEN', 'DELETED'}.
        You can find states' descriptions here https://docs.yandex-team.ru/sandbox/resources#status
    attrs : Resource attributes (system and user defined).
        You can find system attributes' descriptions here https://docs.yandex-team.ru/sandbox/resources#attributes
    owner : Resource owner
    arch  : For what architecture resource was built for
    """

    return resource_type \
        .find(**kwargs) \
        .order(-sdk2.Resource.id) \
        .first()


def find_latest_resource(resource_type, environment, **kwargs):
    """Find latest appropriate released Sandbox resource.

    resource_type : descendant of sdk2.Resource
        Sandbox resource type to search.

    environment : "testing" or "stable"
        Environment for which the search is performed.

        For "stable" search is performed only among resources that were released
        to stable.

        In case of "testing" function searches in both testing and stable and
        chooses resource with greater id.

    **kwargs : additional resource attributes
    """
    stable = get_last_resource(
        resource_type, state='READY', attrs=dict(released='stable', **kwargs))
    if environment == 'stable':
        return stable

    testing = get_last_resource(
        resource_type, state='READY', attrs=dict(released='testing', **kwargs))
    if environment == 'testing':
        if testing is None:
            return stable
        if stable is None:
            return testing
        return max(stable, testing, key=lambda resource: resource.id)
    raise ValueError('environment "{}" not supported'.format(environment))
