import logging
import requests
import time
import copy
import json

from sandbox.sandboxsdk.channel import channel

from requests.packages.urllib3.exceptions import (
    InsecureRequestWarning, InsecurePlatformWarning, SNIMissingWarning)
requests.packages.urllib3.disable_warnings(
    (InsecureRequestWarning, InsecurePlatformWarning, SNIMissingWarning))

_API_URL = "https://platform.yandex-team.ru/api/"
_ACTIVATION_TIMEOUT_SECS = 1200
_DEPLOYMENT_TIMEOUT_SECS = 3600


class QloudClientTimeouts(object):
    def __init__(self, activation_timeout_secs, deployment_timeout_secs):
        self.activation_timeout_secs = activation_timeout_secs
        self.deployment_timeout_secs = deployment_timeout_secs


class QloudUpdateException(RuntimeError):
    pass


class QloudClient:
    def __init__(self, oauth_token, api_url=None, timeouts=None):
        if api_url is None:
            api_url = _API_URL
        if timeouts is None:
            timeouts = QloudClientTimeouts(
                _ACTIVATION_TIMEOUT_SECS, _DEPLOYMENT_TIMEOUT_SECS)
        self.timeouts = timeouts
        self.api_url = api_url
        self.headers = {"Authorization": "OAuth {}".format(oauth_token)}

    @staticmethod
    def _server_call(request_cmd, retries=3, timeout=60):
        # Sometimes Qloud server respond with 502/503 status code (task QLOUD-1983)
        # Try to overcome the issue on client side
        response = None
        for i in xrange(retries):
            try:
                response = request_cmd()
                if 499 < response.status_code < 600:
                    logging.warn("Got unexpected status code from Qloud {}, retry".format(response.status_code))
                    time.sleep(timeout)
                else:
                    break
            except Exception as e:
                logging.warn("Got exception requesting Qloud {}, retry".format(e))
                time.sleep(timeout)
        return response

    def env_dump_url(self, environment_id):
        return self.api_url + "v1/environment/dump/{environment_id}".format(
            environment_id=environment_id)

    def env_upload_url(self):
        return self.api_url + "v1/environment/upload"

    def env_status_url(self, environment_id):
        return self.api_url + "v1/environment/status/{environment_id}".format(
            environment_id=environment_id)

    def dump(self, environment_id):
        res = QloudClient._server_call(lambda: requests.get(
            self.env_dump_url(environment_id),
            headers=self.headers,
            verify=False))  # Sometimes verificantion fails for some reason. See https://st.yandex-team.ru/QLOUDSUPPORT-1366
        if res.status_code != 200:
            raise Exception("Failed to dump environmnet {} with code {}: {}".format(
                environment_id, res.status_code, res.content))
        return res.json()

    def upload(self, env):
        res = QloudClient._server_call(lambda: requests.post(
            self.env_upload_url(), json=env,
            headers=self.headers,
            verify=False))
        if res.status_code != 200:
            raise Exception("Failed to upload environmnet {} with code {}: {}".format(
                env["objectId"], res.status_code, res.content))

    def status(self, environment_id):
        res = QloudClient._server_call(lambda: requests.get(
            self.env_status_url(environment_id),
            headers=self.headers,
            verify=False))
        if res.status_code != 200:
            raise Exception("Failed to get status of environmnet {} with code {}: {}".format(
                environment_id, res.status_code, res.content))
        return json.loads(res.content)

    def wait_deployment(self, environment_id):
        deadline = time.time() + self.timeouts.deployment_timeout_secs
        while True:
            time.sleep(5)
            env_status_resp = self.status(environment_id)
            status = env_status_resp["status"]
            logging.info("Current deployment status {} deadline {}".format(status, deadline))
            stamp = time.time()
            if stamp > deadline:
                raise Exception("Deploy timeout exceeded")
            if status in ["CANCELED", "REMOVED", "FAILED", "DEPLOYED"]:
                break
            elif status in ["ACTIVATING", "DEPLOYING"]:
                deadline = min(stamp + self.timeouts.activation_timeout_secs, deadline)

        if status != "DEPLOYED":
            error_text = "Deployment failed with unexpected status {}".format(env_status_resp["status"])
            if status == "FAILED":
                error_text += ": {}".format(env_status_resp["statusMessage"])
            raise Exception(error_text)


def update_qloud_environment(qloud_environment_id, oauth_token, update_env_cb, on_failure_email, timeouts=None):
    logging.info("Get current environment for {}".format(qloud_environment_id))
    qloud_client = QloudClient(oauth_token, timeouts=timeouts)
    env = qloud_client.dump(qloud_environment_id)
    old_env = copy.deepcopy(env)

    logging.info("Update environment")
    update_env_cb(env)

    def deploy(environment):
        try:
            logging.info("Upload updated environment {} for {}".format(environment, qloud_environment_id))
            qloud_client.upload(environment)
            logging.info("Wait for deployment")
            qloud_client.wait_deployment(qloud_environment_id)
            logging.info("Build deployed")
        except Exception as error:
            logging.exception("Got an exception while update")
            return error
        return None

    update_error = deploy(env)
    if update_error is not None:
        logging.warn("Restoring previous state")
        restore_error = deploy(old_env)
        if restore_error is not None:
            subject = "Both update and restore of {} have failed".format(qloud_environment_id)
            message = "\n\n".join((
                "Got an exception while restoring previous state",
                str(restore_error),
                "Original update error", str(update_error)))
            channel.sandbox.send_email(on_failure_email,
                                       None,
                                       subject,
                                       message)
            raise QloudUpdateException(subject + "\n" + message)
        raise QloudUpdateException("Update of {} has failed, previous state is restored. Update error:\n {}".format(
            qloud_environment_id, update_error))
    logging.info("Finish")


def latest_sandbox_resource_id(type):
    url = "https://sandbox.yandex-team.ru/api/v1.0/resource?type={type}&state=READY&limit=1".format(type=type)
    res = requests.get(url)
    j = res.json()
    item = j["items"][0]
    assert(item["type"] == type)
    return item["id"]
