import contextlib
import json
import logging
import os
import re
import requests
import time
import urllib2
import datetime
from dateutil.parser import parse as datetime_parse

from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia, SvnError
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.vcs.arc import Arc, ResetMode, ArcCommandFailed
from sandbox.projects.maps.common.retry import retry

BUILD_INFO_FILE_NAME = 'build_info.json'
ARCANUM_API_URL = 'https://a.yandex-team.ru/api'
NANNY_API_URL = 'http://nanny.yandex-team.ru'


@contextlib.contextmanager
def nullcontext(enter_result):
    yield enter_result


def ignore_exceptions(f):
    def wrapper(*fargs, **fkwargs):
        try:
            return f(*fargs, **fkwargs)
        except:
            return None
    return wrapper


def write_build_info_file(data_path, arcadia_revision, dataset_version, attributes):
    logging.info("Writing build info to '{}', task ID: {}, revision: {}"
                 .format(data_path, sdk2.Task.current.id, arcadia_revision))
    info = {
        'sandbox_task_id': sdk2.Task.current.id,
        'sandbox_task_type': sdk2.Task.current.type.__name__,
        'sandbox_task_owner': sdk2.Task.current.owner,
        'arcadia_revision': arcadia_revision,
        'dataset_version': dataset_version
    }
    info.update(attributes)

    info_json = json.dumps(info, indent=4)
    file_path = os.path.join(data_path, BUILD_INFO_FILE_NAME)
    with open(file_path, 'w') as info_file:
        info_file.write(info_json)


def execute(obj, command, log_name):
    with sdk2.helpers.ProcessLog(obj, logger=logging.getLogger(log_name)) as process_log:
        return sdk2.helpers.subprocess.Popen(
            command,
            shell=True,
            stdout=process_log.stdout,
            stderr=process_log.stdout,
            cwd=os.getcwd()
        ).wait()


@ignore_exceptions
@retry(tries=3, delay=5)
def get_reviews(auth_token, limit):
    request = urllib2.Request(
        '{}/review/dashboard?view=outgoing&status=pending_review&limit={}'.format(ARCANUM_API_URL, limit),
        headers={"Authorization": "OAuth {}".format(auth_token)})
    return json.load(urllib2.urlopen(request))["reviews"]


@ignore_exceptions
@retry(tries=3, delay=5)
def get_branch(review_id, auth_token):
    request = urllib2.Request(
        '{}/v1/review-requests/{}?fields=vcs(from_branch)'.format(ARCANUM_API_URL, review_id),
        headers={"Authorization": "OAuth {}".format(auth_token)})
    return json.load(urllib2.urlopen(request))["data"]["vcs"]["from_branch"]


def commit_to_arcadia(path, message, user, with_revprop=None):
    try:
        opts = Arcadia.SvnOptions()
        opts.force = True
        opts.depth = 'infinity'
        Arcadia.svn('add', opts=opts, path=path)
        output = Arcadia.commit(path=path, message=message, user=user, with_revprop=with_revprop)
        return output, None, None  # TODO: Parse revision?
    except SvnError as err:
        match = re.search(r'https?://a.yandex-team.ru/(?:arc/)?review/(\d+)', str(err))
        if match:
            return str(err), match.group(0), int(match.group(1))
        else:
            raise


@ignore_exceptions
@retry(tries=3, delay=5)
def get_review_id(revision, auth_token):
    request = urllib2.Request(
        '{}/tree/commit/{}?repo=arc'.format(ARCANUM_API_URL, revision),
        headers={'Authorization': 'OAuth {}'.format(auth_token)})
    message = json.load(urllib2.urlopen(request))['message']
    match = re.search(r'REVIEW:\s+(?P<id>\d+)', message)
    return int(match.group('id')) if match else None


@ignore_exceptions
@retry(tries=3, delay=5)
def add_comment(review, content, auth_token):
    request = urllib2.Request(
        '{}/v1/review-requests/{}/comments'.format(ARCANUM_API_URL, review),
        headers={'Authorization': 'OAuth {}'.format(auth_token),
                 'Content-Type': 'application/json'},
        data=json.dumps({'content': content})
    )
    return urllib2.urlopen(request)


def mount_arcadia(revision=None):
    changeset = 'HEAD'
    if revision:
        changeset = 'r{}'.format(revision)
    return Arc().mount_path('', changeset=changeset, fetch_all=False)


def arc_commit(mount_path, paths, message, description=None, branch=None, publish=True, automerge=False,
               commit_age_threshold=None):
    """
    If path was committed less then commit_age_threshold ago, ignore it
    """
    def is_fresh(path):
        if commit_age_threshold:
            commit_info = arc.log(mount_point=mount_path, path=path, max_count=1, as_dict=True)
            if commit_info:
                commit_time = datetime_parse(commit_info[0]['date']).replace(tzinfo=None)
                return (datetime.datetime.now() - commit_time).total_seconds() < commit_age_threshold

    if description:
        message += "\n" + description
    if branch is None:
        branch = abs(hash(','.join(sorted(path.replace(mount_path, '') for path in paths))))
    branch = 'users/robot-maps-sandbox/{}'.format(branch)
    arc = Arc()
    for path in paths:
        if is_fresh(path):
            logging.warning('skipping path {}, because of time tolerance {} secs'.format(path, commit_age_threshold))
            continue
        arc.add(mount_point=mount_path, path=path)
    arc_status = arc.status(mount_point=mount_path, as_dict=True)
    logging.info('arc_status: {}'.format(arc_status))
    staged = arc_status.get('status', {}).get('staged', [])
    if not staged:  # nothing to commit
        msg = 'Skipping zero diff for commit in "{}"'.format(', '.join(paths))
        logging.info(msg)
        return msg
    arc.checkout(mount_point=mount_path, create_branch=True, branch=branch)
    arc.commit(mount_point=mount_path, message=message)
    arc.push(mount_point=mount_path, upstream=branch, force=True)
    try:
        out = arc.pr_create(mount_point=mount_path, message=message, publish=publish, auto=automerge, no_commits=True)
    except ArcCommandFailed:
        try:
            out = 'Review updated:\n' + arc.pr_status(mount_point=mount_path)
        except ArcCommandFailed:
            out = 'Unable to create or update review'
    arc.reset(mount_point=mount_path, mode=ResetMode.MIXED, path='HEAD~1')  # bring back initial HEAD
    return out


def svn_revision_to_arc_hash(revision, mount_path=None):
    arc = Arc()
    if mount_path:
        init_repo = nullcontext(enter_result=mount_path)
    else:
        init_repo = arc.init_bare()

    with init_repo as mount_path:
        return arc.svn_rev_to_arc_hash(mount_path, revision)


def vcs_info(arcadia_url, arcadia_mount_path):
    url = Arcadia.parse_url(arcadia_url)
    is_arc_repo = arcadia_sdk.is_arc_path(arcadia_url)
    commit_info = arcadia_sdk.mounted_path_svnversion(
        arcadia_mount_path,
        arc_vcs=is_arc_repo
    )
    if is_arc_repo:
        vcs_version = str(commit_info['revision'])
        if not vcs_version.startswith('arc:'):
            # arc trunk
            svn_revision = int(vcs_version)
            arc_commit_hash = svn_revision_to_arc_hash(
                revision=svn_revision,
                mount_path=arcadia_mount_path,
            )
            branch = 'trunk'
        else:
            # arc branch
            svn_revision = None
            arc_commit_hash = vcs_version[len('arc:'):]
            arc_commit_id = url.revision
            branch = None
    else:
        svn_revision = int(commit_info['revision'])
        if url.trunk:
            # svn trunk
            arc_commit_hash = svn_revision_to_arc_hash(svn_revision)
        else:
            # svn branch
            arc_commit_hash = None
        branch = url.branch or url.tag or 'trunk'
    return {
        'svn_revision_ya_package_compatible': str(commit_info['revision']),  # can be either revision or arc:<hash>
        'svn_revision': svn_revision,
        'arc_commit_hash': arc_commit_hash,
        'branch': branch
    }
