# -*- coding: utf-8 -*-

import logging
from collections.abc import Iterator

import yaqutils.misc_helpers as umisc
from facebook_business.adobjects.adaccount import AdAccount
from facebook_business.adobjects.customaudience import CustomAudience
from facebook_business.api import FacebookAdsApi
from facebook_business.api import FacebookSession
from yaqtypes import Token

FACEBOOK_API_VERSION = "v12.0"
UPLOAD_BUCKET_SIZE = 10000


def create_facebook_audience(
        account_id: str,
        name: str,
        description: str,
        facebook_token: Token
) -> CustomAudience:
    session = FacebookSession(access_token=facebook_token)
    api = FacebookAdsApi(session, api_version=FACEBOOK_API_VERSION)

    ad_account = AdAccount(account_id, api=api)

    existing_audiences = ad_account.get_custom_audiences(fields=[CustomAudience.Field.name])
    for existing_audience in existing_audiences:
        if existing_audience[CustomAudience.Field.name] == name:
            return existing_audience

    fields = []
    audience_params = {
        CustomAudience.Field.name: name,
        CustomAudience.Field.subtype: CustomAudience.Subtype.custom,
        CustomAudience.Field.description: description,
        CustomAudience.Field.customer_file_source: CustomAudience.CustomerFileSource.user_provided_only
    }

    facebook_audience = ad_account.create_custom_audience(fields=fields, params=audience_params)
    logging.info(f"Segment {facebook_audience.get_id()} was created at Facebook")

    return facebook_audience


def update_audience(
        audience_from_yt: Iterator[str],
        facebook_audience: CustomAudience,
        id_type: str,
        operator: str
) -> None:
    if id_type == "mobileId":
        schema = CustomAudience.Schema.mobile_advertiser_id
    elif id_type == "hashedPhoneNumber":
        schema = CustomAudience.Schema.email_hash
    elif id_type == "hashedEmail":
        schema = CustomAudience.Schema.phone_hash
    else:
        raise ValueError(f"Unknown id_type: {id_type}")

    for bucket in umisc.split_by_chunks_iter(audience_from_yt, batch_size=UPLOAD_BUCKET_SIZE):
        if operator == "ADD":
            facebook_audience.add_users(schema, bucket)
        else:
            facebook_audience.remove_users(schema, bucket)
        logging.info(f"Bucket with {len(bucket)} records was uploaded to Facebook")
