from sandbox import sdk2
from sandbox.projects.common import binary_task


def load_df(yt_table, column, yql_token, cluster):
    import yt.wrapper as yt

    yt.config["proxy"]["url"] = "{}.yt.yandex.net".format(cluster)
    yt.config["token"] = yql_token

    flow = yt.read_table(yt_table)
    for row in flow:
        yield row[column]


def get_google_client(oauth2_client, client_secret, refresh_token, developer_token, client_customer_id):

    from googleads import adwords, oauth2

    oauth2_client = oauth2.GoogleRefreshTokenClient(oauth2_client, client_secret, refresh_token)
    client = adwords.AdWordsClient(
        developer_token=developer_token,
        oauth2_client=oauth2_client,
        client_customer_id=client_customer_id,
        timeout=30,
    )
    return client


def create_user_list(user_list_service, list_name, id_type):
    if id_type == "mobileId":
        upload_key_type = "MOBILE_ADVERTISING_ID"
    else:
        upload_key_type = "CONTACT_INFO"

    user_list = {
        'xsi_type': 'CrmBasedUserList',
        'name': list_name,
        'description': 'WEEKLY UPLOAD {}'.format(list_name),
        'membershipLifeSpan': 10000,
        'uploadKeyType': upload_key_type,  # MOBILE_ADVERTISING_ID, CONTACT_INFO
    }
    if id_type == "mobileId":
        user_list['appId'] = '313877526'

    operations = [{
        'operator': 'ADD',
        'operand': user_list
    }]
    result = user_list_service.mutate(operations)
    user_list_id = result['value'][0]
    return user_list_id


def mutate_list(user_list_service, user_list_id, ids, id_type, operator):
    members = [{id_type: i} for i in ids]
    mutate_members_operation = {
        'operand': {
            'userListId': user_list_id,
            'membersList': members,
        },
        'operator': operator
    }

    response = user_list_service.mutateMembers([mutate_members_operation])
    if 'userLists' in response:
        for user_list in response['userLists']:
            print('User list with name "%s" and ID "%d" was updated with %d users'
                  % (user_list['name'], user_list['id'], len(members)))


def update_user_list(user_list_service, user_list, audience, id_type, operator):
    batch_size = 20000
    user_list_id = user_list['id']

    ids = []
    for index, row in enumerate(audience, 1):
        ids.append(row)
        if index % batch_size == 0:
            mutate_list(
                user_list_service=user_list_service,
                user_list_id=user_list_id,
                ids=ids,
                id_type=id_type,
                operator=operator
            )
            ids = []

    if len(ids):
        mutate_list(
            user_list_service=user_list_service,
            user_list_id=user_list_id,
            ids=ids,
            id_type=id_type,
            operator=operator
        )


def update_audience(client, audience, id_type, list_name, operator):
    user_list_service = client.GetService('AdwordsUserListService', 'v201809')
    selector = {
        'fields': ['Id', 'Name', 'Status'],
        'predicates': [
            {
                'field': 'Name',
                'operator': 'EQUALS',
                'values': [list_name]
            }
        ]
    }

    ul = user_list_service.get(selector)
    if ul['totalNumEntries'] == 0:
        if operator == "ADD":
            user_list = create_user_list(user_list_service, list_name, id_type)
        else:
            raise Exception("There is not list with name = {} to remove from.".format(list_name))
    elif ul['totalNumEntries'] == 1:
        user_list = ul['entries'][0]
    else:
        raise Exception("More than one user list with Name={}".format(list_name))

    update_user_list(user_list_service, user_list, audience, id_type, operator)


class GoogleAudienceLoader(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Parameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        yt_table = sdk2.parameters.String("YT TABLE", required=True)
        cluster = sdk2.parameters.String("CLUSTER", required=True)
        list_name = sdk2.parameters.String("LIST NAME", required=True)

        with sdk2.parameters.String("Id Type") as id_type:
            id_type.values["MOBILE_ADVERTISING_ID"] = id_type.Value("mobileId", default=True)
            id_type.values["PHONE"] = "hashedPhoneNumber"
            id_type.values["EMAIL"] = "hashedEmail"

        with sdk2.parameters.String("Operator") as operator:
            operator.values["ADD"] = "ADD"
            operator.values["REMOVE"] = "REMOVE"

        with sdk2.parameters.Group("Aut Settings", collapse=False) as aut_settings_group:
            secret_yql_token = sdk2.parameters.YavSecret("YQL TOKEN", required=True)
            client_customer_id = sdk2.parameters.String("CLIENT ID", required=True)
            secret_google_token = sdk2.parameters.YavSecret("GOOGLE TOKEN", required=True)

    def on_execute(self):
        yql_token = self.Parameters.secret_yql_token.data()['nirvana-secret']

        oauth2_client = self.Parameters.secret_google_token.data()['oauth2_client']
        client_secret = self.Parameters.secret_google_token.data()['client_secret']
        refresh_token = self.Parameters.secret_google_token.data()['refresh_token']
        developer_token = self.Parameters.secret_google_token.data()['developer_token']

        client_customer_id = self.Parameters.client_customer_id
        list_name = self.Parameters.list_name
        id_type = self.Parameters.id_type
        operator = self.Parameters.operator

        yt_table = self.Parameters.yt_table
        cluster = self.Parameters.cluster

        audience = load_df(yt_table, id_type, yql_token, cluster)

        adwords_client = get_google_client(
            oauth2_client=oauth2_client,
            client_secret=client_secret,
            refresh_token=refresh_token,
            developer_token=developer_token,
            client_customer_id=client_customer_id
        )
        update_audience(
            client=adwords_client,
            audience=audience,
            list_name=list_name,
            id_type=id_type,
            operator=operator
        )
