import logging

from sandbox import sdk2
from sandbox.sdk2 import yav


def yql_to_df(yt_table, yql_token):
    from yql.api.v1.client import YqlClient

    query = "USE hahn; SELECT * FROM `//home/geo-analytics/mayhem/gma-1123/convs`"

    client = YqlClient(token=yql_token)
    request = client.query(query, syntax_version=1)
    request.run()
    results = request.get_results()

    df = results.full_dataframe

    df["google_click_id"] = df["google_click_id"].apply(lambda x: x.split("#")[0] if "#" in x else x)
    df["google_click_id"] = df["google_click_id"].apply(lambda x: x.encode("ascii", errors="ignore").decode())

    return df


def get_google_client(oauth2_client, client_secret, refresh_token, developer_token):
    from googleads import adwords, oauth2

    oauth2_client = oauth2.GoogleRefreshTokenClient(oauth2_client, client_secret, refresh_token)
    client = adwords.AdWordsClient(
        developer_token=developer_token,
        oauth2_client=oauth2_client,
        client_customer_id="865-686-2627",
        timeout=30,
    )
    return client


def log_response(error, df_vals):
    vals = df_vals.to_dict()

    if not error:
        vals.update({"response": "OK"})
    else:
        vals.update({"response": error})

    return vals


def main(client, click_id, conversion_name, conversion_time, conversion_value):
    offline_conversion_feed_service = client.GetService("OfflineConversionFeedService", version="v201809")

    feed = {
        "conversionName": conversion_name,
        "conversionTime": conversion_time,
        "conversionValue": conversion_value,
        "googleClickId": click_id
    }

    offline_conversion_operation = {
        "operator": "ADD",
        "operand": feed
    }

    offline_conversion_response = offline_conversion_feed_service.mutate([offline_conversion_operation])
    new_feed = offline_conversion_response

    return new_feed


def load_convs(df, adwords_client):
    from datetime import datetime
    import pandas as pd
    from googleads.errors import GoogleAdsServerFault

    logs = pd.DataFrame(columns=list(df.columns) + ["response"])

    for i in range(df.shape[0]):
        click_id = df.iloc[i]["google_click_id"]
        conv_name = df.iloc[i]["conversion_name"]
        conv_time = datetime.strptime(df.iloc[i].conversion_time, "%Y-%m-%dT%H:%M:%S,Europe/Moscow").strftime("%Y%m%d %H%M%S Europe/Moscow")
        conv_val = df.iloc[i]["conversion_value"]

        error = None
        try:
            _ = main(adwords_client, click_id, conv_name, conv_time, conv_val)
        except GoogleAdsServerFault as e:
            error = str(e).split('.')[1].split(' ')[0]
        finally:
            logs = logs.append(log_response(error, df.iloc[i]), ignore_index=True)

    return logs


class GoogleConversionLoader(sdk2.Task):
    def on_create(self):
        self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={
                "target": "maps/geomarketing/GoogleConversionLoader/bin"
            }
        ).first()

    def on_execute(self):
        secret_yql_token = yav.Secret("sec-01f3q8rv815pp44k6fkm8qr6fw")
        secret_google_token = yav.Secret("sec-01f8qef50vh47jk5pmg5kxwny7")

        yql_token = secret_yql_token.data()["nirvana-secret"]

        oauth2_client = secret_google_token.data()["oauth2_client"]
        client_secret = secret_google_token.data()["client_secret"]
        refresh_token = secret_google_token.data()["refresh_token"]
        developer_token = secret_google_token.data()["developer_token"]

        yt_table = self.Parameters.yt_table

        # Read conversions from YT
        convs_df = yql_to_df(yt_table, yql_token)
        logging.info(f"{convs_df.shape[0]} convs were obtained from YT")

        # Create google ads client
        adwords_client = get_google_client(oauth2_client, client_secret, refresh_token, developer_token)

        # Upload conversions
        logs_df = load_convs(convs_df, adwords_client)
        response_ok_count = len(convs_df[convs_df['response'] == 'OK'])
        response_not_ok_count = len(convs_df[convs_df['response'] != 'OK'])
        logging.info(f'Result Success: {response_ok_count}, Error: {response_not_ok_count})')

        return logs_df
