import os

from sandbox import sdk2

from sandbox.sdk2.helpers import ProcessLog, subprocess

from sandbox.projects.maps.common.latest_resources import find_latest_resource
from sandbox.projects.maps.common.juggler_alerts import TaskJugglerReportWithParameters

from sandbox.common.errors import ResourceNotFound, TaskError
from sandbox.common.types import task as ctt


class GeoqResourceWithAutoDiscovery(sdk2.Resource):
    releasable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class GeoqExecutableResourceWithAutoDiscovery(GeoqResourceWithAutoDiscovery):
    executable = True


# these constants are mirroring yandex_environment names
# yet still you can import stuff from Arcadia in the client side methods only, too bad!
STABLE = 'stable'
TESTING = 'testing'
DEVELOPMENT = 'development'


class GeoqBinaryTask(TaskJugglerReportWithParameters):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 32 * 1024  # MiB

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        with sdk2.parameters.Group('Task environment') as environment_parameters:
            with sdk2.parameters.RadioGroup('Environment') as environment:
                environment.values[STABLE] = environment.Value(STABLE)
                environment.values[TESTING] = environment.Value(TESTING)
                environment.values[DEVELOPMENT] = environment.Value(DEVELOPMENT, default=True)

    @property
    def stage_environment(self):
        return self.Parameters.environment

    def check_subtasks(self):
        """Checks all subtasks statuses"""

        if not all(
            task.status == ctt.Status.SUCCESS
            for task in self.find(parent_id=self.id)
        ):
            raise TaskError('Child task failed')

    def fetch_resource(self, resource):
        if not resource:
            raise ResourceNotFound('Resource was not found. Please provide a valid resource.')
        return str(sdk2.ResourceData(resource).path)

    def _prepare_executable(self, executable_resource):
        if issubclass(type(executable_resource), sdk2.Resource):
            return self.fetch_resource(executable_resource)
        return executable_resource

    def run_executable(self, executable_resource, executable_arguments, process_log_name='Executable'):
        """Runs given executable resource

        executable_resource  : sdk2.Resource, executable resource
        executable_arguments : list of the executable arguments
        process_log_name     : name of the executable log on Sandbox
        """

        # ensure every argument is str
        executable_arguments = [str(arg) for arg in executable_arguments]

        executable = self._prepare_executable(executable_resource)
        args = [executable] + executable_arguments

        with ProcessLog(self, process_log_name) as pl:
            return_code = subprocess.call(
                args, stdout=pl.stdout, stderr=pl.stderr)

        if return_code != 0:
            raise TaskError(
                'Executable {} failed with {} return code'.format(
                    args, return_code))

    def _ensure_latest_resources(self):
        """
        Fills all unspecified resource parameters with resource_type=GeoqResourceWithAutoDiscovery with the appropriate resources.

        !!! CAUTION !!!
        This method can be used in the sever-side methods only!
        """

        for name, value in self.Parameters:
            field_resource_type = getattr(type(self.Parameters), name).resource_type
            if not field_resource_type:
                continue

            resource_object = sdk2.Resource[field_resource_type]
            if value is None and issubclass(resource_object, GeoqResourceWithAutoDiscovery):
                assert not self.Parameters.__locked__, (
                    'Looks like you\'re trying to use _ensure_latest_resources in the client-side method. '
                    '`self.Parameters` are locked at this stage.'
                )
                self.Parameters.__setattr__(
                    name, find_latest_resource(resource_object, self.stage_environment))

    def on_save(self):
        super(GeoqBinaryTask, self).on_save()
        if self.stage_environment != 'development' and not self.Requirements.tasks_resource:
            self.Requirements.tasks_resource = find_latest_resource(
                sdk2.service_resources.SandboxTasksBinary, self.stage_environment, task_type=self.type.name)

    def on_prepare(self):
        os.environ.update({
            'ENVIRONMENT_NAME': self.stage_environment,
            # backwards compatability with the hypotheses
            'YENV_TYPE': self.stage_environment,
            'SYNCHROPHAZOTRON_PATH': str(self.synchrophazotron)
        })

    def on_enqueue(self):
        super(GeoqBinaryTask, self).on_enqueue()
        if self.stage_environment != 'development':
            self._ensure_latest_resources()

    def on_execute(self):
        super(GeoqBinaryTask, self).on_execute()
