from collections import defaultdict
from urlparse import urljoin

import numpy as np

import logging
import requests
import json

from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from sandbox import sdk2
from sandbox.common.errors import ResourceNotFound, TaskFailure

from sandbox.projects.maps.geoq.GeoqBinaryTask import GeoqBinaryTask, GeoqExecutableResourceWithAutoDiscovery


STAT_GET_HANDLER = 'https://upload.stat.yandex-team.ru/_api/statreport/json/'

STAT_API_PAYLOAD = {
    'geo_path': '\t10000\t',
    'geo_path__lvl': 0,
    'user_resolved_path': '\t_total_\t',
    'user_resolved_path__lvl': 0,
    'type_path': '\t_total_\t',
    'type_path__lvl': 0,
    'test_id_path': '\t_total_\t',
    'test_id_path__lvl': 0,
    'window_days': 1
}

STAT_PROVIDER = 'source_path'
STAT_PROVIDER_NAME_FIELD = '{}_override_by_dictionary'.format(STAT_PROVIDER)


class StatfaceInvalidResponseError(Exception):
    def __init__(self, response):
        super(StatfaceInvalidResponseError, self).__init__(
            'Received invalid response from Statface with code {}. '
            'Response:\n{}'.format(response.status_code, response.text))


class GeoqHypothesesWatcherConfig(GeoqExecutableResourceWithAutoDiscovery):
    # config example https://paste.yandex-team.ru/1018203

    releasable = True
    executable = False
    releasers = ['MAPS-GEOQ-RELEASERS']


class GeoqHypothesesWatcher(GeoqBinaryTask):
    class Parameters(GeoqBinaryTask.Parameters):

        with sdk2.parameters.Group('Watcher parameters') as watcher_params:
            stat_api_secret = sdk2.parameters.YavSecret(
                'Yav Secret with Yandex.Stat OAuth token',
                required=True)

            json_config = sdk2.parameters.Resource(
                'Monitoring\'s json config',
                resource_type=GeoqHypothesesWatcherConfig,
                required=True)

    def get_monitoring_config(self):
        json_config_id = self.Parameters.json_config
        if json_config_id is None:
            raise ResourceNotFound('Json config was not found')

        json_config = sdk2.ResourceData(json_config_id)
        with open(str(json_config.path)) as f:
            return json.load(f)

    @staticmethod
    def prepare_requests_with_retries(
        retries=3,
        backoff_factor=0.3,
        status_forcelist=(500, 502, 503, 504)
    ):
        retry = Retry(
            total=retries,
            backoff_factor=backoff_factor,
            status_forcelist=status_forcelist,
        )
        adapter = HTTPAdapter(max_retries=retry)

        session = requests.Session()
        session.mount('http://', adapter)
        session.mount('https://', adapter)
        return session

    def request_stat(self, project_name, required_stats, period_distance):
        STAT_API_PAYLOAD.update({
            '_field': required_stats,
            '_period_distance': '{}d'.format(period_distance)
        })

        host = urljoin(STAT_GET_HANDLER, project_name)

        stat_token = self.Parameters.stat_api_secret.value()

        response = self.prepare_requests_with_retries().get(
            host,
            headers={
                'Authorization': 'OAuth {}'.format(stat_token)},
            params=STAT_API_PAYLOAD)

        try:
            return response.json()['values']
        except Exception:
            # I don't know if it is the proper way to reraise an exception
            raise StatfaceInvalidResponseError(response)

    @staticmethod
    def pick_status(value, parameters):
        if parameters.get('CRIT', -np.inf) >= value:
            return 'CRIT', parameters['CRIT']
        if parameters.get('WARN', -np.inf) >= value:
            return 'WARN', parameters['WARN']
        return 'OK', np.inf

    @staticmethod
    def format_juggler_report(provider_report):
        return '{provider}:{stat} - {value:.4f} <= {hit_threshold} ({status})'.format(
            **provider_report)

    def on_execute(self):
        watcher_config = self.get_monitoring_config()
        provider_thresholds = watcher_config['monitorings']

        if not provider_thresholds:
            raise TaskFailure('Monitoring config is empty')

        required_stats = set()
        for stats in provider_thresholds.values():
            for stat in stats.keys():
                required_stats.add(stat)
        required_stats.add(STAT_PROVIDER)

        providers_stats_values = self.request_stat(
            watcher_config['project_name'],
            required_stats,
            watcher_config['period_distance'])

        # provider_stats consists of {
        #     provider_name_1: {
        #         stat_name_1: [stat_value_1, ...],
        #         ...
        #     },
        #     ...
        # }
        provider_stats = defaultdict(lambda: defaultdict(list))

        for stat in providers_stats_values:
            provider_name = stat[STAT_PROVIDER_NAME_FIELD]
            if provider_name not in provider_thresholds:
                continue

            for stat_name in provider_thresholds[provider_name]:
                stat_value = stat.get(stat_name, np.float)
                provider_stats[provider_name][stat_name].append(stat_value)

        status_by_importance = {
            'CRIT': 0,
            'WARN': 1,
            'OK': 2
        }

        metrics_with_status = []
        for provider_name, stat in provider_stats.items():
            logging.info(provider_name)
            for stat_name, stat_values in stat.items():
                threshold = provider_thresholds[provider_name][stat_name]
                none_interpretation = np.float(threshold.get('None', 0.0))
                stat_mean = np.nanmean([
                    (value if value else none_interpretation)
                    for value in stat_values])
                status, hit_threshold = self.pick_status(stat_mean, threshold)
                logging.info(
                    '    %s: %f <= %f (%s)',
                    stat_name, stat_mean, hit_threshold, status)

                if status == 'OK':
                    continue

                metrics_with_status.append({
                    'provider': provider_name,
                    'stat': stat_name,
                    'value': stat_mean,
                    'hit_threshold': hit_threshold,
                    'status': status
                })

        metrics_with_status = sorted(
            metrics_with_status,
            key=lambda x: status_by_importance[x['status']])

        description = (
            '; '.join(map(self.format_juggler_report, metrics_with_status))
            if metrics_with_status else 'Everything is fine.')
        worst_status = (
            metrics_with_status[0]['status']
            if metrics_with_status else 'OK')

        self.send_status_to_juggler(
            service='quality',
            status=worst_status,
            description=description)
