import logging

from datetime import datetime, timedelta

from sandbox import sdk2

import sandbox.common.types.task as ctt
from sandbox.common.errors import TaskError
from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters)

import statface_report

from check_pipeline import CheckPipeline


class GeoqPipelinesHealthCheckerReport(sdk2.Resource):
    executable = False


class GeoqPipelinesHealthCheckerReportUploader(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class GeoqPipelinesHealthChecker(TaskJugglerReportWithParameters):
    class Context(sdk2.Task.Context):
        overall_status = 'Pending'
        overall_description = 'Pending'

        check_pipelines = [
            CheckPipeline(
                name='PERSONAL_POIS_CONVEYOR',
                scheduler=9930,
                no_data_timeout=timedelta(
                    days=1, hours=4).total_seconds()).dump(),
            CheckPipeline(
                name='[STABLE] BUILD_STATIC_POI',
                scheduler=16793,
                no_data_timeout=timedelta(
                    days=1, hours=4).total_seconds()).dump(),
            CheckPipeline(
                name='[TESTING] BUILD_STATIC_POI',
                scheduler=16769,
                no_data_timeout=timedelta(
                    days=1, hours=4).total_seconds()).dump(),
            CheckPipeline(
                name='MAPS_ZERO_SPEED_BANNER_PIPELINE',
                scheduler=18602,
                no_data_timeout=timedelta(
                    days=8).total_seconds()).dump(),
            CheckPipeline(
                name='NEWSFEED_HYPOTHESES_GENERATION',
                scheduler=21659,
                no_data_timeout=timedelta(
                    days=1, hours=12).total_seconds()).dump(),
            CheckPipeline(
                name='POI_METRICS_CALCULATOR',
                scheduler=22543,
                no_data_timeout=timedelta(
                    days=1, hours=4).total_seconds()).dump(),
        ]

    class Parameters(TaskJugglerReportWithParameters.Parameters):

        with sdk2.parameters.Group('Statface parameters') as watcher_params:
            send_statface_report = sdk2.parameters.Bool(
                'Send statface report', default=False)
            use_beta = sdk2.parameters.Bool(
                'Use Statface Beta', default=False)

            stat_api_secret = sdk2.parameters.YavSecret(
                'Yav Secret with Yandex.Stat OAuth token')
            stat_api_secret_key = sdk2.parameters.String(
                'Yav Secret\'s Key')

    def restore_context(self):
        if not self.scheduler:
            return
        previous = list(
            sdk2.Task.find(scheduler=(self.scheduler, -self.scheduler), status=ctt.Status.SUCCESS)
                .order(-sdk2.Task.id)
                .limit(1))
        if previous:
            logging.info('Found previous task with context {}'.format(
                previous[0].Context.check_pipelines))
            self.Context.check_pipelines = previous[0].Context.check_pipelines

    def on_prepare(self):
        self.restore_context()
        self.check_pipelines = [
            CheckPipeline(**task)
            for task in self.Context.check_pipelines
        ]

    @sdk2.header()
    def output_header(self):
        return self.Context.overall_status + '<br>' + self.Context.overall_description.replace('\n', '<br>')

    def format_description(self, overall_description):
        if not overall_description:
            return 'WARN', 'Overall description is empty'

        overall_description = sorted(
            overall_description,
            key=lambda x: x['worst_issue'].priority)
        overall_status = overall_description[0]['worst_issue'].status

        result_description = []
        for description in overall_description:
            result_description.append(
                'Pipeline {pipeline} returned {status}: {issues}'.format(
                    pipeline=description['pipeline'],
                    status=description['worst_issue'].status,
                    issues=map(str, description['issues'])))

        return overall_status, '\n'.join(result_description)

    def send_statface_report(self, overall_statface_report, now):
        if not self.Parameters.send_statface_report:
            return

        if not self.Parameters.stat_api_secret:
            raise TaskError(
                'No Statface Secret was provided')

        stat_token = self.Parameters.stat_api_secret.data().get(
            self.Parameters.stat_api_secret_key)

        if not stat_token:
            raise TaskError(
                'Can\'t find statface token with given name {}'.format(
                    self.Parameters.stat_api_secret_key))

        statface_report.send_statface_report(
            stat_token, overall_statface_report, now, self.Parameters.use_beta)

    def on_execute(self):
        now = datetime.utcnow()

        overall_description = []
        overall_statface_report = []
        for pipeline in self.check_pipelines:
            logging.warning('Checking pipeline %s', pipeline.name)

            pipeline.check(now)

            if pipeline.issues:
                overall_description.append({
                    'pipeline': '{name} ({id})'.format(
                        name=pipeline.name, id=pipeline.scheduler),
                    # issues are sorted by is issue priority
                    'worst_issue': pipeline.issues[0],
                    'issues': pipeline.issues,
                    'check': pipeline
                })

            overall_statface_report.append(
                statface_report.format_report_row(pipeline))

            logging.warning(
                'Pipeline %s checked: %i unfinished, %i successes, %i failures',
                pipeline.name, len(pipeline.unfinished_tasks), len(pipeline.successful_tasks),
                len(pipeline.failed_tasks))

        self.Context.overall_status, self.Context.overall_description = self.format_description(
            overall_description)

        self.send_statface_report(overall_statface_report, now)

    def on_success(self, prev_status):
        self.Context.check_pipelines = [
            check_pipeline.dump()
            for check_pipeline in self.check_pipelines
        ]

        super(GeoqPipelinesHealthChecker, self).on_success(prev_status)
