import copy
import inspect

from sandbox import sdk2
import sandbox.common.types.task as ctt

import issue_types
from utils import duration, timestamp, mean


TASK_WINDOW_SIZE = 16
TASK_FIND_QUERY_SIZE = 32

FINISHED_STATUSES = ctt.Status.Group.FINISH + ctt.Status.Group.BREAK

DURATION_GROW_FRACTION_LIMIT = 1.5


class CheckPipeline:
    # All serializable fields are declared here...
    def __init__(
        self,
        scheduler, no_data_timeout, name=None,
        unfinished_tasks_ids=None,
        successful_tasks_durations=None,
        last_seen_timestamp=0
    ):
        frame = inspect.currentframe()
        args, _, _, values = inspect.getargvalues(frame)

        self._serializable_keys = args[1:]
        for param in self._serializable_keys:
            setattr(self, param, copy.deepcopy(values[param]))

        if not self.successful_tasks_durations:
            self.successful_tasks_durations = []

        # ...and all non-serializable fields are declared here
        self.issues = []

        self.successful_tasks = []
        self.failed_tasks = []
        self.unfinished_tasks = []

    def dump(self):
        return {
            key: getattr(self, key)
            for key in self._serializable_keys
        }

    def check_no_data(self, tasks, now):
        # Checks for juggler style NO_DATA issue and appends it to self.issues
        if not tasks:
            return
        min_duration = min([
            duration(task.updated, now)
            for task in tasks
        ])
        if min_duration > self.no_data_timeout:
            self.issues.append(
                issue_types.NoDataIssue(min_duration))

    def check_duration(self, now, successful_tasks_durations, unfinished_tasks):
        # Checks for hanged tasks and appends description to self.issues (if there are such tasks)
        if len(successful_tasks_durations) < TASK_WINDOW_SIZE:
            return

        mean_duration = mean(successful_tasks_durations)
        for task in unfinished_tasks:
            work_duration = duration(task.created, now)
            if (work_duration / mean_duration > DURATION_GROW_FRACTION_LIMIT):
                self.issues.append(
                    issue_types.LongExecutionTime(
                        task.id, work_duration, mean_duration))

    def split_tasks(self, tasks):
        # Splits tasks into successful, failed and unfinished.
        # If first finished task failed, appends description to self.issues
        unfinished, successful, failed = [], [], []
        for task in tasks:
            if task.status not in FINISHED_STATUSES:
                unfinished.append(task)
            elif task.status in ctt.Status.Group.SUCCEED:
                successful.append(task)
            else:
                if not self.successful_tasks:
                    self.issues.append(
                        issue_types.LastTaskFailure(
                            task.id, task.status))

                failed.append(task)

        return unfinished, successful, failed

    def check(self, now):
        tasks = sdk2.Task.find(scheduler=(self.scheduler, -self.scheduler)) \
                    .limit(TASK_FIND_QUERY_SIZE) \
                    .order(-sdk2.Task.id)
        tasks = list(filter(
            lambda task: timestamp(task.created) > self.last_seen_timestamp,
            tasks))

        if self.unfinished_tasks_ids:
            tasks += list(
                sdk2.Task.find(id=self.unfinished_tasks_ids)
                    .limit(TASK_FIND_QUERY_SIZE)
                    .order(-sdk2.Task.id))

        if not tasks:
            return

        if not self.name:
            # This is serializable field and will be used in the next run
            self.name = str(tasks[0].type)

        self.check_no_data(tasks, now)

        self.unfinished_tasks, self.successful_tasks, self.failed_tasks = (
            self.split_tasks(tasks))

        self.successful_tasks_durations = (
            list(map(
                lambda task: duration(task.created, task.updated),
                self.successful_tasks))
            + self.successful_tasks_durations)[:TASK_WINDOW_SIZE]

        # This is serializable field and will be used in the next run
        self.unfinished_tasks_ids = list(map(
            lambda task: task.id,
            self.unfinished_tasks))

        self.check_duration(
            now, self.successful_tasks_durations, self.unfinished_tasks)

        # This is serializable field and will be used in the next run
        self.last_seen_timestamp = timestamp(tasks[0].created)

        self.issues = sorted(
            self.issues,
            key=lambda x: x.priority)
