import requests
from collections import defaultdict

import sandbox.common.types.task as ctt

from utils import duration, mean


class StatfaceInvalidResponseError(Exception):
    def __init__(self, response):
        super(StatfaceInvalidResponseError, self).__init__(
            'Received invalid response from Statface with code {}. '
            'Response:\n{}'.format(response.status_code, response.text))


def format_report_row(pipeline):
    statface_report_row = {
        'pipeline': '{name} ({id})'.format(
            name=pipeline.name, id=pipeline.scheduler),
        'duration': (
            duration(
                pipeline.successful_tasks[0].created,
                pipeline.successful_tasks[0].updated)
            if pipeline.successful_tasks else None
        ),
        'ma_duration': mean(pipeline.successful_tasks_durations),
        'failure_count': len(pipeline.failed_tasks),
        'successful_count': len(pipeline.successful_tasks)
    }
    failed_statuses_stats = defaultdict(int)
    for task in pipeline.failed_tasks:
        failed_statuses_stats[str(task.status)] += 1
    statface_report_row.update(**failed_statuses_stats)

    return statface_report_row


STATFACE_REPORT_API = 'https://upload.stat.yandex-team.ru/_api/report'
STATFACE_BETA_REPORT_API = 'https://upload.stat-beta.yandex-team.ru/_api/report'

TITLE = 'Geoq Pipelines Health Check'
PATH = 'Maps.Wiki/GEOQ/pipelines_health_check'

SCALE = 'd'

BREAK_GROUP = list(map(lambda x: x.lower(), ctt.Status.Group.BREAK))

USER_CONFIG = {
    'aggregate_method': {
        'duration': 'avg',
        'exception': 'sum',
        'expired': 'sum',
        'failure_count': 'sum',
        'no_res': 'sum',
        'stopped': 'sum',
        'successful_count': 'sum',
        'timeout': 'sum'
    },
    'aggregate_scales': ['w_by_d_avg'],
    'allow_recalculate': 0,
    'autovalues_enabled': 0,
    'dimensions': [
        {'fielddate': 'date'},
        {'pipeline': 'string'}
    ],
    'measures': [
        {'duration': 'number'},
        {'ma_duration': 'number'},
        {'successful_count': 'number'},
        {'failure_count': 'number'},
        {'exception': 'number'},
        {'no_res': 'number'},
        {'timeout': 'number'},
        {'stopped': 'number'},
        {'expired': 'number'}
    ],
    'titles': {
        'duration': 'Pipeline running time in seconds',
        'exception': 'Number of runs with EXCEPTION status',
        'expired': 'Number of runs with EXPIRED status',
        'failure_count': 'Number of failed runs',
        'fielddate': 'Shipping Date',
        'ma_duration': 'Pipeline moving average running time in seconds',
        'no_res': 'Number of runs with NO_RES status',
        'pipeline': 'Tasks',
        'stopped': 'Number of runs with STOPPED status',
        'successful_count': 'Number of successful runs',
        'timeout': 'Number of runs with TIMEOUT status'
    },
    'view_types': {
        'duration': 'Number',
        'exception': 'Number',
        'expired': 'Number',
        'failure_count': 'Number',
        'fielddate': 'Date',
        'ma_duration': 'Number',
        'no_res': 'Number',
        'pipeline': {'allow_in_table': 1, 'type': 'Selector'},
        'stopped': 'Number',
        'successful_count': 'Number',
        'timeout': 'Number'
    }
}


def send_statface_report(stat_token, overall_statface_report, now, use_beta=False):
    statface_api = STATFACE_BETA_REPORT_API if use_beta else STATFACE_REPORT_API

    def prepare_row(row):
        row = {
            key.lower(): value
            for key, value in row.items()
        }
        for status in BREAK_GROUP:
            row[status] = row.get(status, 0)
        row['fielddate'] = now.strftime('%Y-%m-%d')
        return row

    headers = {'Authorization': 'OAuth {}'.format(stat_token)}

    r = requests.post(
        statface_api + '/config/' + PATH,
        headers=headers,
        json={
            'user_config': USER_CONFIG,
            'title': TITLE,
        })

    if r.status_code != 200:
        raise StatfaceInvalidResponseError(r)

    r = requests.post(
        statface_api + '/data/' + PATH,
        params={'scale': SCALE},
        headers=headers,
        json={
            'data': list(map(prepare_row, overall_statface_report))
        })

    if r.status_code != 200:
        raise StatfaceInvalidResponseError(r)
