import os
import json
import multiprocessing

from sandbox import sdk2
from sandbox.projects.maps.geoq.GeoqBinaryTask import GeoqBinaryTask, GeoqExecutableResourceWithAutoDiscovery
from sandbox.projects.maps.common.latest_resources import get_last_resource


class GeoqManoeuvresUploadParametersJson(sdk2.Resource):
    pass


class GeoqManoeuvresPrepareDatasetMetadata(GeoqExecutableResourceWithAutoDiscovery):
    pass


class GeoqManoeuvresPrepare(GeoqExecutableResourceWithAutoDiscovery):
    pass


class GeoqManoeuvresUpload(GeoqExecutableResourceWithAutoDiscovery):
    pass


class GeoqManoeuvresHypotheses(GeoqBinaryTask):
    class Parameters(GeoqBinaryTask.Parameters):
        with sdk2.parameters.Group('Executable') as executable_parameters:
            prepare_metadata_executable = sdk2.parameters.Resource(
                'Prepare dataset metadata executable',
                resource_type=GeoqManoeuvresPrepareDatasetMetadata)
            prepare_executable = sdk2.parameters.Resource(
                'Prepare & Nirvana executable', resource_type=GeoqManoeuvresPrepare)
            upload_executable = sdk2.parameters.Resource(
                'Upload executable', resource_type=GeoqManoeuvresUpload)

        with sdk2.parameters.Group('Tokens') as token_parameters:
            yt_token = sdk2.parameters.YavSecret('YT OAuth token', required=True)
            nirvana_token = sdk2.parameters.YavSecret('Nirvana OAuth token', required=True)
            tvm_token = sdk2.parameters.YavSecret('TVM token')

        with sdk2.parameters.Group('Prepare Dataset Metadata stage') as prepare_dataset_metadata_parameters:
            graph_version = sdk2.parameters.String(
                'Graph versions', description='(the most recent is used by default)')
            start_date = sdk2.parameters.String(
                'Start date', description=r'in format %Y-%m-%d')
            date_count = sdk2.parameters.Integer(
                'Date count', default=7)

        with sdk2.parameters.Group('Prepare dataset stage') as prepare_dataset_parameters:
            dataset_static_map_api_key = sdk2.parameters.YavSecret(
                'Static Map API key',
                description='key for dataset',
                required=True)
            dataset_static_map_api_secret = sdk2.parameters.YavSecret(
                'Static Map API secret',
                description='key for dataset',
                required=True)

            table_weight = sdk2.parameters.Float(
                'Dataset weight', default=0.01)
            threads_count = sdk2.parameters.Integer(
                'Threads count')
            rps_limit = sdk2.parameters.Float(
                'RPS limit', default=500.0)
            nirvana_workflow = sdk2.parameters.String(
                'Nirvana workflow ID')

        with sdk2.parameters.Group('Upload') as upload_parameters:
            upload_static_map_api_key = sdk2.parameters.YavSecret(
                'Static Map API key',
                description='key for descriptions')
            upload_static_map_api_secret = sdk2.parameters.YavSecret(
                'Static Map API secret',
                description='key for descriptions')

            dry_run = sdk2.parameters.Bool(
                'Dry run', description='No hypotheses will be uploaded', default=False)
            use_tvm = sdk2.parameters.Bool(
                'Use tvm', default=False)

    def run_prepare_dataset_metadata_stage(self):
        package_tree = self.fetch_resource(self.Parameters.prepare_metadata_executable)

        # inside of `prepare_metadata_executable` is an Arcadia root-like structure with two directories
        # `maps/geoq/...` and `maps/analyzer/...`.
        # at some point prepare_dataset_1 will try to call some executable from maps/analyzer
        # for that reason, we're changing current working directory to this root folder...
        previous_root = os.getcwd()
        package_root = os.path.dirname(package_tree)
        os.chdir(package_root)

        arguments = ['--date-count', self.Parameters.date_count]
        if self.Parameters.start_date:
            arguments += ['--start-date', self.Parameters.start_date]
        if self.Parameters.graph_version:
            arguments += ['--graph-version', self.Parameters.graph_version]

        self.run_executable(
            'maps/geoq/hypotheses/manoeuvres/prepare_dataset_1/prepare_dataset_1', arguments, 'prepare_dataset_metadata')

        # ...and then, changing it back to the previous one
        os.chdir(previous_root)

    def run_prepare_dataset_stage(self):
        os.environ.update({
            'STATIC_MAP_TOKEN': self.Parameters.dataset_static_map_api_key.value(),
            'STATIC_MAP_SECRET': self.Parameters.dataset_static_map_api_secret.value()
        })

        arguments = [
            '--weight', self.Parameters.table_weight,
            '--threads', self.Parameters.threads_count or multiprocessing.cpu_count(),
            '--rps-limit', self.Parameters.rps_limit
        ]
        if self.Parameters.nirvana_workflow:
            arguments += ['--nirvana', self.Parameters.nirvana_workflow]

        self.run_executable(self.Parameters.prepare_executable, arguments, 'prepare_dataset')

    def get_last_upload_parameters(self):
        last_resource = get_last_resource(GeoqManoeuvresUploadParametersJson)
        with open(self.fetch_resource(last_resource), 'r') as f:
            return json.load(f)

    def run_upload_stage(self):
        last_upload_parameters = self.get_last_upload_parameters()

        arguments = ['--table', last_upload_parameters['table']]

        if self.Parameters.upload_static_map_api_key and self.Parameters.upload_static_map_api_secret:
            os.environ.update({
                'STATIC_MAP_TOKEN': self.Parameters.upload_static_map_api_key.value(),
                'STATIC_MAP_SECRET': self.Parameters.upload_static_map_api_secret.value(),
            })

        if self.Parameters.use_tvm:
            os.environ.update({
                'TVM_TOKEN': self.Parameters.tvm_token.value()
            })
            arguments.append('--use-tvm')

        if self.Parameters.dry_run:
            arguments.append('--dry-run')

        self.run_executable(self.Parameters.upload_executable, arguments, 'upload')

    def on_execute(self):
        os.environ.update({
            'YT_TOKEN': self.Parameters.yt_token.value(),
            'NIRVANA_TOKEN': self.Parameters.nirvana_token.value()
        })

        self.run_prepare_dataset_metadata_stage()
        self.run_prepare_dataset_stage()
        self.run_upload_stage()
