import os

from sandbox import sdk2
from sandbox.projects.maps.geoq.GeoqBinaryTask import GeoqBinaryTask, GeoqExecutableResourceWithAutoDiscovery


class GeoqOsmHypothesesExecutable(GeoqExecutableResourceWithAutoDiscovery):
    pass


class GeoqOsmHypotheses(GeoqBinaryTask):
    class Parameters(GeoqBinaryTask.Parameters):
        with sdk2.parameters.Group('Executable') as executable_parameters:
            executable = sdk2.parameters.Resource(
                'OSM hypotheses executable', resource_type=GeoqOsmHypothesesExecutable)

        with sdk2.parameters.Group('Tokens') as token_parameters:
            yt_token = sdk2.parameters.YavSecret('YT OAuth token', required=True)
            yql_token = sdk2.parameters.YavSecret('YQL OAuth token', required=True)
            tvm_token = sdk2.parameters.YavSecret('TVM token')

        with sdk2.parameters.Group('Pipeline parameters') as pipeline_parameters:
            input_tables = sdk2.parameters.List(
                'Input tables', value_type=sdk2.parameters.String, default=[])
            proximity = sdk2.parameters.Float(
                'Road graph snap proximity in meters', default=50.0)
            threshold = sdk2.parameters.Float(
                'Join threshold in meters', default=300.0)
            exclude_types = sdk2.parameters.List(
                'Road types to exclude', value_type=sdk2.parameters.String, default=[])
            exclude_operations = sdk2.parameters.List(
                'OSM operations to exclude', value_type=sdk2.parameters.String, default=[])
            store_locally = sdk2.parameters.Bool(
                'Store locally or upload result to YT')
            output_table = sdk2.parameters.String('Output table')

        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            memory_limit = sdk2.parameters.Integer(
                'Memory limit in GB', default=8)
            job_count = sdk2.parameters.Integer(
                'YT job count', default=32)

        with sdk2.parameters.Group('Upload parameters') as upload_parameters:
            source_name = sdk2.parameters.String(
                'Hypotheses source name', default='geoq-roads-d1', required=True)
            dry_run = sdk2.parameters.Bool(
                'Dry run', description='No hypotheses will be uploaded', default=False)
            use_tvm = sdk2.parameters.Bool(
                'Use tvm', default=False)

    def on_execute(self):
        os.environ.update({
            'YQL_TOKEN': self.Parameters.yql_token.value(),
            'YT_TOKEN': self.Parameters.yt_token.value()
        })

        arguments = [
            # pipeline parameters
            '--proximity', self.Parameters.proximity,
            '--threshold', self.Parameters.threshold,
            '--verbose',
            # YT parameters
            '--memory-limit', self.Parameters.memory_limit,
            '--job-count', self.Parameters.job_count,
            '--source-name', self.Parameters.source_name
        ]

        if self.Parameters.dry_run:
            arguments.append('--dry-run')

        if self.Parameters.use_tvm:
            os.environ.update({
                'TVM_TOKEN': self.Parameters.tvm_token.value()
            })
            arguments.append('--use-tvm')

        if self.Parameters.exclude_types:
            arguments += ['--exclude-types'] + self.Parameters.exclude_types

        if self.Parameters.exclude_operations:
            arguments += ['--exclude-operations'] + self.Parameters.exclude_operations

        # store locally
        if self.Parameters.store_locally:
            arguments.append('--store-locally')

        # input tables
        if self.Parameters.input_tables:
            arguments += ['--input-tables'] + self.Parameters.input_tables

        # output table
        if self.Parameters.output_table:
            arguments += [
                '--output-tables', self.Parameters.output_table
            ]

        self.run_executable(self.Parameters.executable, arguments)
