import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
from sandbox.common.errors import TaskFailure, TaskNotEnqueued
from urlparse import urljoin

from sandbox import sdk2
from sandbox.common.types import task as ctt


CI_TOKEN = 'sec-01fddc7kv32yhbt4vzvnr8j1ta#ci.token'
NANNY_CI_TOKEN = 'sec-01fddc7kv32yhbt4vzvnr8j1ta#nanny_ci.token'


class ServiceType:
    Nanny = 'nanny'
    Garden = 'garden'
    Sandbox = 'sandbox'


class Status:
    Ready = 'ready'
    Broken = 'broken'


class SedemMachineReleaseBuild(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 1024  # 1GB
        ram = 2048  # 2GB

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):

        commit_identity = sdk2.parameters.String('Arc commit hash', required=True)
        service_name = sdk2.parameters.String('Sedem service name', required=True)
        service_path = sdk2.parameters.String('Path to service arcadia dir', required=True)
        service_type = sdk2.parameters.String('Type of service', required=True)
        use_testing_sedem = sdk2.parameters.Bool('Use testing sedem for notifications')

    def send_notification(self, status, error=None):
        # type: (str, str) -> None
        session = requests.Session()
        adapter = HTTPAdapter(max_retries=Retry(10, backoff_factor=10))
        session.mount('http://', adapter)
        session.mount('https://', adapter)
        params = {
            'operation_id': self.id,
            'status': status,
        }
        if error:
            params['error'] = error
        url = (
            'http://core-sedem-machine.testing.maps.yandex.net'
            if self.Parameters.use_testing_sedem else
            'http://core-sedem-machine.maps.yandex.net'
        )
        response = session.post(
            url=urljoin(url, '/v1/events/build'),
            params=params,
            headers={'host': 'core-sedem.maps.yandex.net'}
        )
        self.set_info('status={}\nresponse: {}'.format(response.status_code, str(response.content)))

    def on_execute(self):
        with self.memoize_stage.build(commit_on_entrance=False):
            self.start_build()
        build_status = self.build_status()
        self.set_ttl(build_status)
        self.report_build(build_status)

    def start_build(self):
        service_type = self.Parameters.service_type.lower()
        if service_type == ServiceType.Nanny:
            build_task_id = self.enqueue_maps_docker()
        elif service_type == ServiceType.Garden:
            build_task_id = self.enqueue_garden_module()
        elif service_type == ServiceType.Sandbox:
            build_task_id = self.enqueue_build_binary_task()
        else:
            raise TaskFailure('Unknown service type: {}'.format(service_type))
        raise sdk2.WaitTask(build_task_id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def set_ttl(self, build_status):
        is_build_failed = bool(build_status != ctt.Status.SUCCESS)
        if is_build_failed:
            return
        service_type = self.Parameters.service_type.lower()
        if service_type == ServiceType.Nanny:
            resource_type = 'YA_PACKAGE_RESOURCE'
        elif service_type == ServiceType.Garden:
            resource_type = 'YT_GARDEN_MODULE_BINARY_RESOURCE'
        elif service_type == ServiceType.Sandbox:
            resource_type = 'SANDBOX_TASKS_BINARY'
        else:
            raise TaskFailure('Unknown service type: {}'.format(service_type))
        build_resource = self.find_build_resource(resource_type)
        build_resource.ttl = 'inf'
        build_resource.backup_task = True

    def report_build(self, build_status):
        is_build_failed = bool(build_status != ctt.Status.SUCCESS)
        if is_build_failed:
            task_id = self.Context.build_task_id
            task_url = 'https://sandbox.yandex-team.ru/task/{}/view'.format(task_id)
            self.send_notification(
                status=Status.Broken,
                error='Task {} ended with status {}'.format(task_url, build_status)
            )
            raise TaskFailure('Failed child build task: {}'.format(task_url))
        else:
            self.send_notification(status=Status.Ready)

    def enqueue_maps_docker(self):
        build_target = '{}/docker/pkg.json'.format(self.Parameters.service_path)
        task = self.server.task({
            'template_alias': 'MAPS_CORE_BUILD_DOCKER_STABLE',
            'children': True,
            'description': self.Parameters.description,
            'owner': 'MAPS-CI',
            'custom_fields': [
                {'name': 'arcadia_url', 'value': 'arcadia-arc:/#{}'.format(self.Parameters.commit_identity)},
                {'name': 'target', 'value': build_target},
                {'name': 'sedem_service_name', 'value': self.Parameters.service_name},
                {'name': 'arc_token', 'value': CI_TOKEN},
                {'name': 'yt_token', 'value': CI_TOKEN},
                {'name': 'docker_token', 'value': NANNY_CI_TOKEN},
                {'name': 'use_testing_sedem', 'value': self.Parameters.use_testing_sedem},
            ],
            'priority': self.Parameters.priority,
        })
        task_id = task['id']
        self.Context.build_task_id = task_id
        result = self.server.batch.tasks.start.update(task_id)[0]  # api returns list of results per task
        if result.get('status') != 'SUCCESS':
            raise TaskNotEnqueued('Task #{} is not enqueued: {}'.format(task_id, result.get('message')))
        return task_id

    def enqueue_garden_module(self):
        module_path = self.Parameters.service_path
        build_target = '{}/bin'.format(module_path)
        _, module_name = module_path.rsplit('/', 1)  # maxsplit=1
        task = self.server.task({
            'template_alias': 'MAPS_CORE_GARDEN_BUILD_MODULE_STABLE',
            'children': True,
            'description': self.Parameters.description,
            'owner': 'MAPS-CI',
            'custom_fields': [
                {'name': 'arcadia_url', 'value': 'arcadia-arc:/#{}'.format(self.Parameters.commit_identity)},
                {'name': 'target', 'value': build_target},
                {'name': 'module_name', 'value': module_name},
                {'name': 'arc_token', 'value': CI_TOKEN},
                {'name': 'yt_token', 'value': CI_TOKEN},
                {'name': 'use_testing_sedem', 'value': self.Parameters.use_testing_sedem},
            ],
            'priority': self.Parameters.priority,
        })
        task_id = task['id']
        self.Context.build_task_id = task_id
        result = self.server.batch.tasks.start.update(task_id)[0]  # api returns list of results per task
        if result.get('status') != 'SUCCESS':
            raise TaskNotEnqueued('Task #{} is not enqueued: {}'.format(task_id, result.get('message')))
        return task_id

    def enqueue_build_binary_task(self):
        build_target = '{}/task'.format(self.Parameters.service_path)
        task = self.server.task({
            'template_alias': 'MAPS_CORE_BUILD_BINARY_TASK_STABLE',
            'children': True,
            'description': self.Parameters.description,
            'owner': 'MAPS-CI',
            'custom_fields': [
                {'name': 'arcadia_url', 'value': 'arcadia-arc:/#{}'.format(self.Parameters.commit_identity)},
                {'name': 'target', 'value': build_target},
                {'name': 'sedem_service_name', 'value': self.Parameters.service_name},
                {'name': 'arc_token', 'value': CI_TOKEN},
                {'name': 'yt_token', 'value': CI_TOKEN},
                {'name': 'use_testing_sedem', 'value': self.Parameters.use_testing_sedem},
            ],
            'priority': self.Parameters.priority,
        })
        task_id = task['id']
        self.Context.build_task_id = task_id
        result = self.server.batch.tasks.start.update(task_id)[0]  # api returns list of results per task
        if result.get('status') != 'SUCCESS':
            raise TaskNotEnqueued('Task #{} is not enqueued: {}'.format(task_id, result.get('message')))
        return task_id

    def build_status(self):
        task_id = self.Context.build_task_id
        return self.server.task[task_id].read()['status']

    def find_build_resource(self, resource_type):
        task_id = self.Context.build_task_id

        resource = sdk2.Resource.find(
            task_id=task_id,
            type=resource_type,
            limit=1,
        ).first()
        if resource:
            return resource

        raise TaskFailure(
            'Cannot find resource {} in task {}'.format(
                resource_type, task_id
            )
        )
