import requests

from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
from sandbox import sdk2
from sandbox.projects.common.vcs.arc import Arc, ArcCommandFailed
from urlparse import urljoin


def obtain_ci_token():
    return sdk2.yav.Secret('sec-01fddc7kv32yhbt4vzvnr8j1ta').data()['ci.token']


class Status:
    Ready = 'ready'
    Broken = 'broken'


class SedemMachineReleaseHotfix(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 1024  # 1GB
        ram = 2048  # 2GB

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):

        target_branch = sdk2.parameters.String('Target branch name (to apply cherry-pick)')
        commit_id = sdk2.parameters.String('Commit id to apply (in form r<SVN revision> or commit hash)',
                                           default_value='trunk')
        new_tag = sdk2.parameters.String('Tag name for updated branch', required=True)
        notification_vhost = sdk2.parameters.String('Vhost to use for requests', default='core-sedem.maps.yandex.net')
        notification_fqdn = sdk2.parameters.String('FQDN to send result notification', required=True)

    @property
    def target_branch(self):
        return self.Parameters.target_branch

    @property
    def new_tag(self):
        return self.Parameters.new_tag

    def send_notification(self, status, error=None):
        # type: (str, str) -> None
        if self.Parameters.notification_fqdn:
            session = requests.Session()
            adapter = HTTPAdapter(max_retries=Retry(10, backoff_factor=10))
            session.mount('http://', adapter)
            session.mount('https://', adapter)
            params = {
                'operation_id': self.id,
                'status': status,
            }
            if error:
                params['error'] = error
            url = 'http://{}'.format(self.Parameters.notification_fqdn)
            response = session.post(
                url=urljoin(url, '/v1/events/hotfix'),
                params=params,
                headers={'host': self.Parameters.notification_vhost}
            )
            self.set_info('status={}\nresponse: {}'.format(response.status_code, str(response.content)))

    def on_execute(self):
        try:
            arc = Arc(arc_oauth_token=obtain_ci_token())
            commit = self.Parameters.commit_id
            if commit.isdigit():
                commit = 'r{}'.format(commit)
            with arc.mount_path(path=None, changeset=self.target_branch, fetch_all=False) as mount_point:
                arc.cherry_pick(mount_point=mount_point, commit=commit, add_commit_name=True)
                arc.push(mount_point=mount_point)
                arc.push(mount_point=mount_point, refspecs=[('HEAD', self.new_tag)])
            self.send_notification(status=Status.Ready)
        except (ArcCommandFailed, AssertionError, Exception):
            task_url = 'https://sandbox.yandex-team.ru/task/{}/view'.format(self.id)
            self.send_notification(
                status=Status.Broken,
                error='Task {} failed'.format(task_url),
            )
            raise
