import os
import tarfile
import logging
import tempfile
import urllib
import base64

from sandbox import sdk2
from sandbox.sandboxsdk import svn
import sandbox.common.types.client as ctc
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.vcs.arc import Arc
from sandbox.projects.maps.common.retry import retry
import sandbox.projects.common.arcadia.sdk as arcadia_sdk

_KEYCHAIN_NAME = 'mobile-signing'
_CODESIGN_PATH = os.path.join(os.sep, 'usr', 'bin', 'codesign')

_PROVISIONING_PROFILE_URL_TEMPLATE = 'https://bitbucket.browser.yandex-team.ru/projects/MT/repos/mobile-ios-provisioning-profiles/raw/AdHoc/Generic_Yandex_In_House_Profile_2014.mobileprovision'
_PROVISIONING_PROFILE_DIR = os.path.join(os.path.expanduser('~'), 'Library', 'MobileDevice', 'Provisioning Profiles')
_PROVISIONING_PROFILE_FILENAME_TEMPLATE = 'Generic_Yandex_In_House_Profile_2014.mobileprovision'

class ThrowingUrlOpener(urllib.FancyURLopener):
    def http_error_default(self, req, fp, code, msg, hdrs):
        urllib.URLopener.http_error_default(self, req, fp, code, msg, hdrs)

class FailedRunBinaryFromArc(sdk2.Task):

    _bundle_task_type = 'MAPS_MOBILE_YA_PACKAGE_DARWIN'

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.CUSTOM_MAPS_MOBILE_DARWIN & ctc.Tag.USER_MAPS_MOBILE_DARWIN

    class Parameters(sdk2.Task.Parameters):
        arcadia_patch = sdk2.parameters.String('Apply patch', required=False)
        signing_certificate = sdk2.parameters.YavSecret(
                    'Yav secret with signing certificate',
                    required=True
                    )
        keychain_password = sdk2.parameters.YavSecret(
                    'Yav secret with maps mobile keychain password',
                    required = True
                    )

    def _checkout_arcadia_src(self):
        arc = Arc()
        return arc.mount_path(path=None, changeset=None, allow_root=True, fetch_all=True)
        # return arcadia_sdk.mount_arc_path('arcadia:/arc/trunk/arcadia@7570086', use_arc_instead_of_aapi=True)

    def _extract_resource(self, resource, path):
        tgz_path = str(sdk2.ResourceData(resource).path)
        with tarfile.open(tgz_path, 'r:gz') as tgz:
            tgz.extractall(path)

    def _get_developer_env(self):
        developer_env = os.environ
        developer_env['DEVELOPER_DIR'] = os.path.join(
            '/Applications',
            'Xcode11.1.app',
            'Contents',
            'Developer'
            )

    def _pod_install(self, arcadia_path):
        pod_env = self._get_developer_env()
        app_path = os.path.join(arcadia_path, 'maps', 'mobile', 'apps', 'TempAppForTestSandbox')
        os.chdir(app_path)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("pod_install")) as pl:
            sp.Popen(['pod', 'install'],
                env=pod_env,
                stdout=pl.stdout,
                stderr=sp.STDOUT).wait()
        return app_path

    @retry(tries=3, delay=10, backoff=2)
    def _retrieve_provisioning_profile(self):
        if not os.path.exists(_PROVISIONING_PROFILE_DIR):
            os.makedirs(_PROVISIONING_PROFILE_DIR)

        ThrowingUrlOpener().retrieve(
            _PROVISIONING_PROFILE_URL_TEMPLATE,
            os.path.join(_PROVISIONING_PROFILE_DIR, _PROVISIONING_PROFILE_FILENAME_TEMPLATE)
            )

    def _run_security(self, args):
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('security')) as pl:
            sp.check_call(
                ['security'] + args,
                stdout=pl.stdout,
                stderr=pl.stdout
                )

    def _set_up_keychain(self):
        keychain_password = self.Parameters.keychain_password.data()['password']
        certificate_file = os.path.abspath('certificate.p12')
        with open(certificate_file, 'wb') as f:
            f.write(base64.b64decode(self.Parameters.signing_certificate.data()['certificate-p12']))

        self._run_security(['create-keychain', '-p', keychain_password, _KEYCHAIN_NAME])
        self._run_security(['unlock-keychain', '-p', keychain_password, _KEYCHAIN_NAME])
        self._run_security([
            'import', certificate_file,
            '-k', _KEYCHAIN_NAME,
            '-P', self.Parameters.signing_certificate.data()['certificate-password'],
            '-T', _CODESIGN_PATH])
        self._run_security(['set-key-partition-list', '-S', 'apple-tool:,apple:', '-k', keychain_password, _KEYCHAIN_NAME])
        self._run_security(['set-keychain-settings', '-lu', '-t', '3600', _KEYCHAIN_NAME])
        self._run_security(['list-keychains', '-s', _KEYCHAIN_NAME])

    def on_execute(self):
        with self._checkout_arcadia_src() as arcadia_path:
            if self.Parameters.arcadia_patch:
                svn.Arcadia.apply_patch(arcadia_path, self.Parameters.arcadia_patch, tempfile.mkdtemp())

            app_path = self._pod_install(arcadia_path)
            self._retrieve_provisioning_profile()
            self._set_up_keychain()

            args = ['xcodebuild', 'DEBUG_INFORMATION_FORMAT=dwarf-with-dsym',
                    '-configuration', 'Debug', '-scheme', 'TempAppForTestSandbox',
                    '-sdk', 'iphoneos',  '-archivePath', os.path.abspath('build') + '/device',
                    '-workspace', os.path.join(app_path, 'TempAppForTestSandbox.xcworkspace'),
                    '-allowProvisioningUpdates',
                    'CODE_SIGN_IDENTITY=iPhone Distribution: Yandex LLC',
                    'PROVISIONING_PROFILE_SPECIFIER=EK7Z26L6D4/Generic Yandex In House Profile 2014',
                    'DEVELOPMENT_TEAM=EK7Z26L6D4', ]

            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("xcodebuild")) as pl:
                sp.check_call(
                    args,
                    env=self._get_developer_env(),
                    cwd=app_path,
                    stdout=pl.stdout,
                    stderr=pl.stdout
                )





