import json
import requests
import os
import subprocess
import functools
import yaml

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.maps.common.ecstatic_bin import MapsEcstaticToolMixin
from sandbox.projects.maps.common.retry import retry
from sandbox.sandboxsdk import environments


DEVELOPER_DASHBOARD_REQUEST_URL = "http://apikeys.yandex.net:8666/api/v2/project_service_link_export?_page_size=-1"
MAPS_MOBILE_PREFIX = 'maps_mobile_'

TVM_IDS = {
    'stable': 2010070,
    'testing': 2010072,
    'datatesting': 2010250,
    'load': 2010260,
}
TVM_SECRETS = {
    'stable': 'sec-01dq7m8pnmyf9ys2907qkt0h3v',
    'testing': 'sec-01dq7m8pw5x1pygrdvs3b1g76q',
    'datatesting': 'sec-01dq7m9b7x36b21skgwhwag7xq',
    'load': 'sec-01dq7m9cfn7gan6vv64jsc7z10',
}

def _client_cmp(client1, client2):
    if 'key' in client1 and 'key' in client2:
        return cmp(client1['key'], client2['key'])
    if 'key' in client1 and 'key' not in client2:
        return -1
    if 'key' not in client1 and 'key' in client2:
        return 1
    return cmp(client1['app_id'], client2['app_id'])


def _custom_retry(exceptions=Exception):
    return retry(exceptions=exceptions, tries=5, delay=30, max_delay=120, backoff=2)


def _fill_prefixes(limits):
    return list({
        key[len(MAPS_MOBILE_PREFIX):].split('@')[0]
        for key in limits.keys()
        if key.startswith(MAPS_MOBILE_PREFIX + '/')
    })


@_custom_retry(exceptions=sdk2.svn.SvnError)
def _fetch_config(config_url):
    return sdk2.svn.Arcadia.cat(config_url)


@_custom_retry(exceptions=requests.exceptions.RequestException)
def _get_developer_dashboard_keys(token):
    response = requests.get(DEVELOPER_DASHBOARD_REQUEST_URL, headers={"X-Service-Token": token})
    response.raise_for_status()
    return response.json()


def _parse_developer_dashboard_client(entry, key):
    client_key = key['key']
    client_type = 'free' if entry['tariff'] == 'mapkit_free' else 'commercial'
    client_app_ids = key.get('custom_params', {}).get('application_ids')
    if client_app_ids is None:
        return {'key': client_key, 'type': client_type}
    return {'key': client_key, 'type': client_type, 'app_ids': client_app_ids}


def _parse_developer_dashboard_clients(key_data):
    clients = []
    for entry in key_data['data']:
        for key in entry['keys']:
            if key['active']:
                clients.append(_parse_developer_dashboard_client(entry, key))
    return clients


def _client_id(client):
    if 'key' in client:
        return client['key']
    if 'app_id' in client:
        return client['app_id']
    raise TaskError('No client id field')


# We need to merge client lists prioritizing client info from config over Dashboard
def _merge_clients(config_clients, developer_dashboard_clients):
    merged_list = []
    id_set = set()
    for client_list in [config_clients, developer_dashboard_clients]:
        new_client_list = [client for client in client_list if not _client_id(client) in id_set]
        new_ids = set([_client_id(client) for client in new_client_list])
        merged_list.extend(new_client_list)
        id_set = id_set.union(new_ids)
    return merged_list


class MapsMobileAccessUpdater(MapsEcstaticToolMixin, sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):

        tvm_environment = sdk2.parameters.String("TVM environment", required=True)
        ecstatic_environment = sdk2.parameters.String("Ecstatic environment", required=True)
        ecstatic_dataset = sdk2.parameters.String("Ecstatic dataset", required=True)
        ecstatic_branch = sdk2.parameters.String("Dataset branch", required=True)

        app_whitelist_clients_config_file = sdk2.parameters.ArcadiaUrl('White list of clients config path', required=True)
        app_blacklist_clients_config_file = sdk2.parameters.ArcadiaUrl('Black list of clients config path', required=True)
        rate_limiter_config_file = sdk2.parameters.ArcadiaUrl('Rate-limiter config path', required=True)

        use_developer_dashboard = sdk2.parameters.Bool('Use API keys from Developer Dashboard', required=True)
        with use_developer_dashboard.value[True]:
            developer_dashboard_token_name = sdk2.parameters.String(
                'Developer Dashboard service token',
                required=True
            )

    @_custom_retry(exceptions=subprocess.CalledProcessError)
    def _upload_ecstatic(self, dir_name, version):
        self.ecstatic(self.Parameters.ecstatic_environment,
             [
                 'upload',
                 '{}={}'.format(self.Parameters.ecstatic_dataset, version),
                 dir_name,
                 '+' + self.Parameters.ecstatic_branch
             ],
             tvm_id=TVM_IDS.get(self.Parameters.tvm_environment),
             tvm_secret_id=TVM_SECRETS.get(self.Parameters.tvm_environment),
             )

    def _update_dataset(self, settings):
        dataset_dir = 'dataset'
        os.makedirs(dataset_dir)
        with open(os.path.join(dataset_dir, 'access-config.json'), 'w') as f:
            json.dump(settings, f, indent=4, sort_keys=True)
        version = str(self.id)

        try:
            self._upload_ecstatic(dataset_dir, version)

        except subprocess.CalledProcessError as e:
            self.set_info(e.output)
            raise TaskError('Ecstatic returned ' + str(e.returncode))

    def _fetch_developer_dashboard_token(self):
        token = sdk2.Vault.data(self.Parameters.developer_dashboard_token_name)
        return token

    def _fetch_developer_dashboard_clients(self):
        token = self._fetch_developer_dashboard_token()
        key_data = _get_developer_dashboard_keys(token)
        clients = _parse_developer_dashboard_clients(key_data)
        return clients

    def on_execute(self):
        limits = yaml.load(_fetch_config(self.Parameters.rate_limiter_config_file),
                           Loader=yaml.SafeLoader)
        config_clients = json.loads(_fetch_config(self.Parameters.app_whitelist_clients_config_file))
        if self.Parameters.use_developer_dashboard:
            developer_dashboard_clients = self._fetch_developer_dashboard_clients()
            clients = _merge_clients(config_clients, developer_dashboard_clients)
        else:
            clients = config_clients
        clients = sorted(clients, key=functools.cmp_to_key(_client_cmp))

        prefixes = _fill_prefixes(limits)

        blacklist = json.loads(_fetch_config(self.Parameters.app_blacklist_clients_config_file))

        settings = {'clients': clients, 'blacklist': blacklist, 'prefixes': prefixes}

        self._update_dataset(settings)
