import datetime
import json
import logging
import os
import re
import requests

from requests.adapters import HTTPAdapter
from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.common.testenv_client import TEClient
from sandbox.projects.maps.common.retry import retry
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2 import yav
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.common import decorators
from sandbox.projects.common import requests_wrapper

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.projects.maps.mobile.utils.testpalm_constants as tpc

_USER_NAME = 'robot-mapkit-ci'
_SSH_SECRET_NAME = 'ssh_key'
_RELEASES_DIRECTORY = 'maps-mobile-releases'
_MOBILE_PATH = os.path.join('maps', 'mobile')
_ISSUE_REGEX = re.compile('[A-Z]+-\d+')
_REVIEW_REGEX = re.compile('REVIEW: \d+')
_TE_TOKEN_SECRET_ID = 'sec-01e3q5djeyyh51anssr125dmt6'
_ST_TOKEN_SECRET_ID = 'sec-01dgcxcnqvd362skd1rde57kkf'
_ST_URL_TEMPLATE = 'https://st.yandex-team.ru/{}'
_ST_DESCRIPTION_TEMPLATE = '''branch: {branch_path}
((https://beta-testenv.yandex-team.ru/project/{te_db_name}/timeline TestEnv))
((https://a.yandex-team.ru/arc/branches/{branch_path}/arcadia Arcanum))
(({testpalm_url} TestPalm))
%%ya clone --branch=branches/{branch_path} {branch_path}/arcadia%%

<{{Changes
{changes}
}}>'''
_ST_ISSUE_TEMPLATE_ID = '4315'

_TRUNK_TE_DB_NAME = 'maps-mobile'
_RELEASE_TE_DB_TEMPLATE = 'maps-mobile-{}'

_EXTRA_NEW_RELEASE_TE_TASKS = {
    'MAPS_MOBILE_BUMP_MAPKIT_IN_NAVIKIT': rm_const.TestFrequency.CHECK_EACH_REVISION,
}

_EXTRA_STOPPED_RELEASE_TE_TASKS = [
    'MAPS_MOBILE_BUILD_ANDROID_AAR_DATASYNC_APP',
    'MAPS_MOBILE_BUILD_ANDROID_AAR_MRC_APP',
    'MAPS_MOBILE_BUILD_ANDROID_AAR_PUSH_APP',
    # MAPS_MOBILE_BUILD_ANDROID_AAR_TEST_APP wasn't stopped

    'MAPS_MOBILE_BUILD_ANDROID_APP_DATASYNC_APP_RELEASE',
    'MAPS_MOBILE_BUILD_ANDROID_APP_DATASYNC_APP_TESTING',
    'MAPS_MOBILE_BUILD_ANDROID_APP_MRC_APP_RELEASE',
    'MAPS_MOBILE_BUILD_ANDROID_APP_MRC_APP_TESTING',
    'MAPS_MOBILE_BUILD_ANDROID_APP_PUSH_APP_RELEASE',
    'MAPS_MOBILE_BUILD_ANDROID_APP_PUSH_APP_TESTING',
    'MAPS_MOBILE_BUILD_ANDROID_APP_TEST_APP_RELEASE',
    'MAPS_MOBILE_BUILD_ANDROID_APP_TEST_APP_RELEASE_WITH_CRASHLYTICS',
    'MAPS_MOBILE_BUILD_ANDROID_APP_TEST_APP_TESTING',

    'MAPS_MOBILE_BUILD_ANDROID_RESOURCE_PROVIDER_AUTOMOTIVE',
    'MAPS_MOBILE_BUILD_ANDROID_RESOURCE_PROVIDER_TRANSPORT',

    'MAPS_MOBILE_BUILD_IOS_APP_DATASYNC_APP_RELEASE',
    'MAPS_MOBILE_BUILD_IOS_APP_DATASYNC_APP_TESTING',
    'MAPS_MOBILE_BUILD_IOS_APP_PUSH_APP_RELEASE',
    'MAPS_MOBILE_BUILD_IOS_APP_PUSH_APP_TESTING',
    'MAPS_MOBILE_BUILD_IOS_APP_TEST_APP_RELEASE',
    'MAPS_MOBILE_BUILD_IOS_APP_TEST_APP_TESTING',

    'MAPS_MOBILE_BUILD_IOS_FRAMEWORK_DATASYNC_APP',
    'MAPS_MOBILE_BUILD_IOS_FRAMEWORK_PUSH_APP',
    'MAPS_MOBILE_BUILD_IOS_FRAMEWORK_TEST_APP',

    'MAPS_MOBILE_BUILD_IOS_RESOURCE_BUNDLE_AUTOMOTIVE',
    'MAPS_MOBILE_BUILD_IOS_RESOURCE_BUNDLE_TRANSPORT',

    'MAPS_MOBILE_BUILD_IOS_RESOURCE_PROVIDER_AUTOMOTIVE',
    'MAPS_MOBILE_BUILD_IOS_RESOURCE_PROVIDER_TRANSPORT',

    'MAPS_MOBILE_PALMSYNC_SYNCHRONIZE',
    'MAPS_MOBILE_PALMSYNC_VALIDATE',

    'MAPS_MOBILE_UPLOAD_ANDROID_APP_DATASYNC_APP_RELEASE',
    'MAPS_MOBILE_UPLOAD_ANDROID_APP_DATASYNC_APP_TESTING',
    'MAPS_MOBILE_UPLOAD_ANDROID_APP_MRC_APP_RELEASE',
    'MAPS_MOBILE_UPLOAD_ANDROID_APP_MRC_APP_TESTING',
    'MAPS_MOBILE_UPLOAD_ANDROID_APP_PUSH_APP_RELEASE',
    'MAPS_MOBILE_UPLOAD_ANDROID_APP_PUSH_APP_TESTING',
    'MAPS_MOBILE_UPLOAD_ANDROID_APP_TEST_APP_RELEASE',
    'MAPS_MOBILE_UPLOAD_ANDROID_APP_TEST_APP_TESTING',
    'MAPS_MOBILE_UPLOAD_ANDROID_CRASHLYTICS_TEST_APP',

    'MAPS_MOBILE_UPLOAD_IOS_APP_DATASYNC_APP_RELEASE',
    'MAPS_MOBILE_UPLOAD_IOS_APP_DATASYNC_APP_TESTING',
    'MAPS_MOBILE_UPLOAD_IOS_APP_PUSH_APP_RELEASE',
    'MAPS_MOBILE_UPLOAD_IOS_APP_PUSH_APP_TESTING',
    'MAPS_MOBILE_UPLOAD_IOS_APP_TEST_APP_RELEASE',
    'MAPS_MOBILE_UPLOAD_IOS_APP_TEST_APP_TESTING',
]

_TESTPALM_SCHEME = 'https://'
_TESTPALM_CLONE_URL_TEMPLATE = _TESTPALM_SCHEME + \
    'testpalm-api.yandex-team.ru/projects/' \
    '{source_project}/clone/{destination_project}' \
    '?title=MapKit%20(Release-{branch_name})&cloneTestcases=true' \
    '&cloneTestRuns=true&cloneVersions=true'
_TESTPALM_RELEASE_BRANCH_PROJECT_URL_TEMPLATE = _TESTPALM_SCHEME + \
    'testpalm.yandex-team.ru/' + \
    tpc.RELEASE_BRANCH_TESTPALM_PROJECT_TEMPLATE
_MAPKIT_RELEASE_ST_TAG = 'mapkit_release_{}'


class ExtendedTEClient(TEClient):
    START_JOB_HANDLER = "handlers/startTest"

    @decorators.retries(3, delay=5, exceptions=(requests_wrapper.ServerError, ))
    def start_test(self, db, job_name):
        url = rm_const.Urls.TESTENV + self.START_JOB_HANDLER
        response = requests_wrapper.get(
            url,
            params={
                "database": db,
                "test_name": job_name
            },
            headers={"Authorization": "OAuth {}".format(self._oauth_token)}
        )
        requests_wrapper.check_status_code(response)
        logging.info("Job %s was started successfully", job_name)


def _release_branch_path(branch_name):
    return '{}/{}'.format(_RELEASES_DIRECTORY, branch_name)


def _release_branch_url(branch_name, path=''):
    return Arcadia.branch_url(_release_branch_path(branch_name), path=path)


def _previous_branch_name(branch_name):
    release_names = Arcadia.list(
        os.path.join(
            Arcadia.ARCADIA_BASE_URL,
            'branches',
            'maps-mobile-releases'
            ),
        as_list=True
        )
    logging.info("Release names: {}".format(release_names))
    branch_index = release_names.index(branch_name + os.path.sep)
    if branch_index != 0:
        return release_names[branch_index - 1][:-1]
    return None


def _folder_commits(branch_name, folder, stop_on_copy, revision_from, revision_to):
    return {commit['revision'] : commit
            for commit in Arcadia.log(
                _release_branch_url(
                    branch_name,
                    path=os.path.join('maps', 'mobile', folder)
                    ),
                stop_on_copy=stop_on_copy,
                revision_from=revision_from,
                revision_to=revision_to
                )
            }


def _mobile_commits(branch_name, stop_on_copy=False, revision_from=None, revision_to=None):
    commits = {}
    for folder in ['apps', 'libs']:
        commits.update(_folder_commits(
            branch_name, folder, stop_on_copy, revision_from, revision_to))
    return sorted(commits.values(), key=lambda x: x['revision'])


def _create_release_branch(branch_name):
    Arcadia.copy(
        Arcadia.trunk_url(),
        _release_branch_url(branch_name),
        "Creating new maps-mobile-releases branch",
        user=_USER_NAME,
        parents=True
        )


def _extract_by_regex(commits, regex):
    return {issue
        for commit in commits
        for issue in regex.findall(commit['msg'])
        }


def _extract_issues(commits):
    return _extract_by_regex(commits, _ISSUE_REGEX)


def _extract_reviews(commits):
    return _extract_by_regex(commits, _REVIEW_REGEX)


def _compose_changelog(branch_name):
    previous_branch_name = _previous_branch_name(branch_name)
    if not previous_branch_name:
        return None
    previous_branch_commits = _mobile_commits(
        previous_branch_name,
        stop_on_copy=True
        )
    previous_branch_reviews = _extract_reviews(previous_branch_commits)
    new_commits = _mobile_commits(
        branch_name,
        revision_from=previous_branch_commits[0]['revision'],
        revision_to='HEAD'
        )
    logging.info('Commits: {}'.format(new_commits))
    new_commits = [commit
        for commit in new_commits
        if (not _extract_reviews([commit])) or  _extract_reviews([commit]) - previous_branch_reviews
        ]
    logging.info('Filtered Commits: {}'.format(new_commits))
    new_issues = _extract_issues(new_commits)
    logging.info('Issues: {}'.format(new_issues))
    changelog = '\n'.join(sorted(new_issues))
    return changelog


@retry(
    exceptions=requests.exceptions.RequestException,
    tries=5,
    delay=30,
    max_delay=120,
    backoff=2
    )
def retry_post_request(url, headers={}):
    return requests.post(
        url=url,
        headers=headers,
        verify=False,
        timeout=30)


class MapsMobileBranchRelease(sdk2.Task):
    ''' Task for building mobile apps '''

    class Requirements(sdk2.Task.Requirements):
        # This fixes connecting to TestPalm servers
        dns = ctm.DnsType.DNS64

        environments = (
            PipEnvironment('yandex-tracker-client', '1.9', custom_parameters=['--upgrade-strategy only-if-needed']),
            PipEnvironment('startrek-client', '2.5', custom_parameters=['--upgrade-strategy only-if-needed']),
        )
        client_tags = ctc.Tag.LINUX_XENIAL

    class Parameters(sdk2.Task.Parameters):
        testpalm_token = sdk2.parameters.YavSecret(
            'TestPalm oauth token of an owner of mapkit_test project',
            required=True)
        branch_name = sdk2.parameters.String('Explicit branch name', required=False)
        only_create_branch_and_testenv = sdk2.parameters.Bool('Only create branch & clone testenv db', default=False, required=False)

    def _clone_testpalm_project(self, branch_name):
        source_project = "mapkit_test"
        destination_project = tpc.RELEASE_BRANCH_TESTPALM_PROJECT_TEMPLATE.format(branch_name=branch_name)
        url = _TESTPALM_CLONE_URL_TEMPLATE.format(
            source_project=source_project,
            destination_project=destination_project,
            branch_name=branch_name)
        oauth = 'OAuth {}'.format(self.Parameters.testpalm_token.data()['testpalm_token'])

        response = retry_post_request(
            url,
            headers={'Authorization': oauth})

        result_info_string = 'TestPalm project cloning response: {code}\n{data}'.format(
                code=response.status_code,
                data=json.dumps(response.json(), indent=4)
                     if 'application/json' in response.headers['content-type']
                     else response.text())

        logging.info(result_info_string)
        if response.status_code in [200, 409]:
            return

        self.set_info(result_info_string)
        raise TaskError('error: {code} is an unexpected status code! Url: {url}'.format(
            code=response.status_code,
            url=url))

    def _clone_testenv_db(self, branch_name):
        new_db_name = _RELEASE_TE_DB_TEMPLATE.format(branch_name)
        token = yav.Secret(_TE_TOKEN_SECRET_ID).data()['testenv_token']
        te_client = ExtendedTEClient(token)
        start_revision = Arcadia.log(
            _release_branch_url(branch_name),
            stop_on_copy=True
            )[-1]['revision']

        params = {
            'source_db_name': _TRUNK_TE_DB_NAME,
            'new_db_name': new_db_name,
            'svn_server': 'arcadia.yandex.ru/arc',
            'svn_path': 'branches/{}'.format(_release_branch_path(branch_name)),
            'start_revision': start_revision,
            'new_task_owner': self.owner,
            'created_by': self.author,
            'clone_owners': '1',
            'add_owners': self.author,
            'start_new_db': '1'
        }
        te_client.clone_db(params)

        for task, frequency in _EXTRA_NEW_RELEASE_TE_TASKS.items():
            te_client.add_test(new_db_name, task, start_revision, frequency)
        for task in _EXTRA_STOPPED_RELEASE_TE_TASKS:
            te_client.start_test(new_db_name, task)

    def _create_stratrek_issue(self, branch_name, changelog):
        from startrek_client import Startrek

        token = yav.Secret(_ST_TOKEN_SECRET_ID).data()['startrek_token']
        st_client = Startrek(token=token, useragent=_USER_NAME)

        description = _ST_DESCRIPTION_TEMPLATE.format(
            branch_path=_release_branch_path(branch_name),
            te_db_name=_RELEASE_TE_DB_TEMPLATE.format(branch_name),
            changes=changelog,
            testpalm_url=_TESTPALM_RELEASE_BRANCH_PROJECT_URL_TEMPLATE.format(branch_name=branch_name)
            )
        template = st_client.issue_templates.get(
            key=_ST_ISSUE_TEMPLATE_ID,
            queue='MAPSMOBCORE',
            )
        followers = [follower.id for follower in template.fieldTemplates['followers']]
        issue = st_client.issues.create(
            queue='MAPSMOBCORE',
            assignee=self.author,
            followers=followers,
            tags=['release', _MAPKIT_RELEASE_ST_TAG.format(branch_name)],
            components='mapkit',
            summary='Maps mobile release {}'.format(branch_name),
            description=description
            )
        return issue.key

    def on_execute(self):
        # assert that startrek_client can be imported in the beginning of task execution
        from startrek_client import Startrek

        with sdk2.ssh.Key(self, _USER_NAME, _SSH_SECRET_NAME):
            branch_name = self.Parameters.branch_name or datetime.datetime.now().strftime('%Y%m%d%H')
            _create_release_branch(branch_name)
            self._clone_testenv_db(branch_name)
            if self.Parameters.only_create_branch_and_testenv:
                return
            self._clone_testpalm_project(branch_name)
            changelog = _compose_changelog(branch_name)
            issue_key = self._create_stratrek_issue(branch_name, changelog)
            self.set_info(_ST_URL_TEMPLATE.format(issue_key))
