import os
import logging
import base64
import shutil

from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.projects.maps.mobile.MapsMobileBuildPlatformProject import MapsMobileBuildPlatformProject
from sandbox.projects.maps.mobile.utils.resource_helpers import extract_resource
from sandbox.projects.maps.mobile.utils.subprocess_helpers import check_call_wrapper
from sandbox.projects.maps.mobile.utils.arcadia_url_helpers import is_branch_up_to_date

GRADLE_PROPERTY_PREFIX = 'ORG_GRADLE_PROJECT_'


class MapsMobileBuildAndroidProject(MapsMobileBuildPlatformProject):
    ''' Task for building Android projects '''

    _bundle_task_type = 'MAPS_MOBILE_YA_PACKAGE_LINUX_XENIAL'

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL

    class Parameters(MapsMobileBuildPlatformProject.Parameters):
        with sdk2.parameters.Group('Project build') as project_block:
            jdk_resource = sdk2.parameters.Resource('JDK resource', required=True)
            android_sdk_resource = sdk2.parameters.Resource('Android SDK resource', required=True)
            gradle_resource = sdk2.parameters.Resource('Gradle resource', required=True)
            maven_repo_resource = sdk2.parameters.Resource('Maven repository resource', required=True)
            custom_build_command = sdk2.parameters.String('Custom build command', required=False)

        with sdk2.parameters.Group('Release information') as release_block:
            group_id = sdk2.parameters.String('Group id')
            artifact_id = sdk2.parameters.String('Artifact name')
            artifact_version = sdk2.parameters.String('Artifact version')

        with sdk2.parameters.Group('Signing') as signing_block:
            keystore = sdk2.parameters.YavSecret(
                'Yav secret with android signing keystore',
                required=True
                )

    def _extract_and_find_aar_dir(self):
        extract_resource(self.Parameters.output_bundle_resource, self._bundle_dir)
        for root, _, files in os.walk(self._bundle_dir):
            for name in files:
                if '.aar' in name:
                    return root
        return None

    def _set_up_structure(self, project_src_path):
        self._bundle_dir = os.path.join('build', 'aars')
        self._jdk_dir = os.path.join('build', 'jdk')
        self._android_sdk_dir = os.path.join('build', 'android_sdk')
        self._gradle_dir = os.path.join('build', 'gradle')
        self._maven_repo_dir = os.path.join('build', 'repo')
        keystore_dir = os.path.join('build', 'keystore')
        self._keystore_file = os.path.join(keystore_dir, 'android.keystore')
        self._maps_mobile_aar_dir = self._extract_and_find_aar_dir()
        extract_resource(self.Parameters.jdk_resource, self._jdk_dir, gzip=False)
        extract_resource(self.Parameters.android_sdk_resource, self._android_sdk_dir)
        extract_resource(self.Parameters.gradle_resource, self._gradle_dir)
        extract_resource(self.Parameters.maven_repo_resource, self._maven_repo_dir)

        os.makedirs(keystore_dir)
        with open(self._keystore_file, 'wb') as f:
            f.write(base64.b64decode(self.Parameters.keystore.data()['keystore']))

    def _run_build(self, project_src_path):
        os.environ['USE_CRASHLYTICS'] = '1'

        assemble_command = self.Parameters.custom_build_command
        if not assemble_command:
            assemble_command = 'assemble' + self.Parameters.build_variant.capitalize()
        keystore_data = self.Parameters.keystore.data()
        properties = {
            'localRepositoryPath': os.path.abspath(self._maven_repo_dir),
            'aarPaths': os.path.abspath(self._maps_mobile_aar_dir),
            'keystorePath': os.path.abspath(self._keystore_file),
            'keystorePass': keystore_data['keystore_pass'],
            'keyAlias': keystore_data['key_alias'],
            'keyPass': keystore_data['key_pass'],
            'versionCode': str(self.id),
            # For arc revision is part of self.Parameters.arcadia_url
            'versionName': self.Parameters.arcadia_revision or self.Parameters.arcadia_url.split('#')[-1],
            'universalApk': '1',
            'disableMapsMobileAarCheck': '1'
            }
        if self.Parameters.group_id and self.Parameters.artifact_id and self.Parameters.artifact_version:
            properties.update({
                'groupId': self.Parameters.group_id,
                'artifactId': self.Parameters.artifact_id,
                'artifactVersion': self.Parameters.artifact_version,
            })
        if is_branch_up_to_date(self.Parameters.arcadia_url, self._FIRST_BRANCH_WITH_ICON_VERSION_WRITER):
            properties.update({
                'iconVersionWriterPath': os.path.dirname(self._icon_version_writer_path),
            })
        gradle_env = os.environ
        gradle_env.update(
            {GRADLE_PROPERTY_PREFIX + prop: value for (prop, value) in properties.items()})
        gradle_env.update({
            'JAVA_HOME': os.path.abspath(self._jdk_dir),
            'ANDROID_HOME': os.path.join(os.path.abspath(self._android_sdk_dir), 'android_sdk')
            })

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("gradle_app_build")) as pl:
            check_call_wrapper(
                '{} --no-daemon {}'.format(
                    os.path.abspath(os.path.join(self._gradle_dir, 'bin', 'gradle')),
                    assemble_command
                    ),
                cwd=project_src_path,
                env=gradle_env,
                shell=True,
                stdout=pl.stdout,
                stderr=pl.stdout
                )
