import os

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.common import build
from sandbox.projects.common.constants import YA_MAKE_FORCE_BUILD_SYSTEM

from sandbox.projects.maps.mobile.MapsMobileMakePlatformArtifactsBase \
        import MapsMobileMakePlatformArtifactsBase
from sandbox.projects.maps.mobile.MapsMobileResources import MapsMobileFrameworkZip
from sandbox.projects.maps.mobile.utils.resource_helpers import (
    extract_resource,
    make_resource,
    apply_navi_export_parameters,
    get_ya_package_resource,
    )
from sandbox.projects.maps.mobile.utils.subtask_runner import SubtaskRunner
from sandbox.projects.maps.mobile.utils.arcadia_url_helpers \
    import is_branch_up_to_date, _FIRST_BRANCH_WITH_STATIC_XCFRAMEWORK, _FIRST_BRANCH_WITH_M1_SIMULATOR_BUILD
from sandbox.projects.maps.mobile.utils.yt_store_parameters import yt_store_parameters
from sandbox.projects.maps.mobile.utils.zip_helpers import pack_into_zip_file



_FRAMEWORK_ZIP_TTL = 2
_YA_PACKAGE_TASK_TYPE = 'MAPS_MOBILE_YA_PACKAGE_DARWIN'
_TEMP_RESOURCES_DIR = '/ios_packager_temp_resources'
_LIBRARIES_WITHOUT_IOS_HEADERS = {
    'mrc',
}
_LIBRARIES_NOT_USING_VULKAN = {
    'auth',
    'datasync',
}
_FIRST_BRANCH_WITH_UNITED_IOS_PACKAGER = '20210309'  # todo replace with real branch MAPSMOBCORE-11739
_FIRST_BRANCH_WITH_IDL_IOS_HEADERS_IN_MRC = '2021110317'
_FIRST_BRANCH_WITH_IOS_HEADERS_IN_MRC = '2022012020'


class MapsMobileBuildFramework(MapsMobileMakePlatformArtifactsBase):
    ''' Task for building iOS framework. '''

    class Parameters(MapsMobileMakePlatformArtifactsBase.Parameters):
        platform = 'ios'

        framework_name = sdk2.parameters.String(
                'Framework name',
                default='YandexMapsMobile',
                required=True,
                )
        ya_make_arcadia_dirname = sdk2.parameters.String(
                'Arcadia dirname to recursive_library ya.make',
                default='maps/mobile/bundle/recursive_library',
                required=True,
                )
        is_dynamic = sdk2.parameters.Bool('Dynamic pkg.json')

        with is_dynamic.value[False]:
            mangle_symbols = sdk2.parameters.Bool('Mangle symbols', default=True)

        with sdk2.parameters.Output:
            framework_tar_gz = sdk2.parameters.Resource('Framework tar.gz')
            framework_zip = sdk2.parameters.Resource('Framework zip')
            deb_pkg_json = sdk2.parameters.Resource('Debian package second pkg.json')
            debian_package = sdk2.parameters.Resource(
                'Debian package',
                description='A .tar.gz containing a .deb',
            )

    def on_execute(self):
        super(MapsMobileBuildFramework, self).on_execute()
        with self.memoize_stage.make_framework_zip:
            self._make_framework_zip()

    def _build_with_ya_package(self):
        subtask_type = _YA_PACKAGE_TASK_TYPE
        with self.memoize_stage['run_{}'.format(subtask_type)]:
            subtask_parameters = {
                build.parameters.ArcadiaUrl.name: self.Parameters.arcadia_url,
                build.parameters.ForceVCSInfoUpdate.name: True,
                build.parameters.UseArcadiaApiFuse.name: True,
                build.parameters.UseArcInsteadOfArcadiaApi.name: True,
                build.parameters.CheckoutParameter.name: True,
                build.parameters.BuildSystem.name: YA_MAKE_FORCE_BUILD_SYSTEM,
                build.YaPackage.AdhocPackagesParameter.name: [self._pkg_json],
                build.YaPackage.PackageTypeParameter.name: 'tarball',
                build.YaPackage.PublishPackageParameter.name: False,
                build.YaPackage.SaveBuildOutputParameter.name: True,
                build.parameters.YaTimeout.name: 12 * 60 * 60,
                'kill_timeout': 12 * 60 * 60,
            }
            subtask_parameters.update(yt_store_parameters())
            subtask_requirements = {
                'disk_space': 20 * 1024  # MiB
            }
            SubtaskRunner(self).run(subtask_type, subtask_parameters, subtask_requirements)
        with self.memoize_stage.get_build_framework_resource:
            subtask = SubtaskRunner(self).find_task(subtask_type)
            SubtaskRunner.require_successful(subtask)
            resource = get_ya_package_resource(subtask)
            if self.Parameters.is_navi:
                apply_navi_export_parameters(resource)
            self.Parameters.framework_tar_gz = resource
            self.Context.framework_artifacts = resource.id

    def _building_pkg_json_build(self):
        build = {}
        for arch in self._architectures:
            build_key = "build_{arch}".format(arch=arch)
            platform_flags = [
                ('NO_DEBUGINFO', 'yes'),
            ]
            if self.Parameters.is_dynamic:
                platform_flags += [
                    ('MAPS_MOBILE_EXPORT_OBJC_API', 'yes'),
                ]
            if self.Parameters.is_dynamic and set(self.Parameters.libraries) - _LIBRARIES_NOT_USING_VULKAN:
                platform_flags += [
                    ('EXPORT_MOLTENVK', 'yes'),
                ]
            build[build_key] = self._build_item(
                    arch=arch,
                    target=self.Parameters.ya_make_arcadia_dirname,
                    platform_flags=platform_flags)
        build_tools_targets = [os.path.dirname(self._ios_packager_binary_path())]
        if self.Parameters.is_navi:
            build_tools_targets.append("maps/mobile/tools/ya-package-mv")
        build["build_tools"] = {
            "targets": build_tools_targets
        }
        return build

    def _building_pkg_json_data(self):
        data = super(MapsMobileBuildFramework, self)._building_pkg_json_data()
        data += self._ios_packager_temp_libs_data()
        data += self._ios_packager_temp_headers_data()
        data += self._ios_packager_temp_resources_data()
        if self.Parameters.is_navi and not self.Parameters.is_dynamic:
            data += self._empty_dir_for_mangled_symbols_data()
            data += self._mangler_data()
        return data

    def _ios_packager_binary_path(self):
        if not is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_UNITED_IOS_PACKAGER) and self.Parameters.is_dynamic:
            return "maps/mobile/tools/ios-packager/dynamic/dynamic-ios-packager"
        return "maps/mobile/tools/ios-packager/ios-packager"

    def _ios_packager_temp_libs_data(self):
        return [
                   {
                       "source": {
                           "type": "BUILD_OUTPUT",
                           "path": os.path.join(
                                           self.Parameters.ya_make_arcadia_dirname,
                                           self._library_basename(),
                                           ),
                           "build_key": "build_{arch}".format(arch=arch),
                        },
                        "destination": {
                            "path": "/" + self._temp_library_basename(arch),
                        }
                   }
                   for arch in self._architectures
               ]

    def _temp_library_basename(self, arch=None):
        return (
            'bundle{dot_arch}.{extension}'
            .format(
                dot_arch=(('.' + arch) if arch else ''),
                extension=('dylib' if self.Parameters.is_dynamic else 'a')
            )
        )

    def _library_basename(self):
        return 'libmaps-mobile.dylib' if self.Parameters.is_dynamic else 'libMapsMobile.a'

    def _ios_packager_temp_headers_data(self):
        if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_IDL_IOS_HEADERS_IN_MRC):
            idl_header_libraries = self._idl_header_libraries
        else:
            idl_header_libraries = self._idl_header_libraries - _LIBRARIES_WITHOUT_IOS_HEADERS

        libraries_without_ios_headers = _LIBRARIES_WITHOUT_IOS_HEADERS
        if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_IOS_HEADERS_IN_MRC):
            libraries_without_ios_headers -= {'mrc'}

        return [
                   {
                       "source": {
                           "type": "ARCADIA",
                           "path": "maps/mobile",
                           "files": [
                               "libs/{lib}/*/ios/include/*.h".format(lib=lib)
                               for lib in set(self._libraries) - libraries_without_ios_headers
                           ]
                       },
                       "destination": {
                           "path": "/ios_packager_temp_headers/"
                       }
                   },
                   {
                       "source": {
                           "type": "BUILD_OUTPUT",
                           "build_key": "build_x86_64",
                           "path": "maps/mobile",
                           "files": [
                               "libs/{lib}/*/idl/include/{framework_name}/*.h".format(
                                       lib=lib,
                                       framework_name=self.Parameters.framework_name,
                                       )
                               for lib in idl_header_libraries
                           ]
                       },
                       "destination": {
                           "path": "/ios_packager_temp_headers/{framework_name}/".format(
                                           framework_name=self.Parameters.framework_name,
                                           )
                       }
                   }
               ]

    def _ios_packager_temp_resources_data(self):
        ios_res_directories = self._directories('ios/res')
        ios_packager_temp_resources = [
            {
                "source": {
                    "type": "ARCADIA",
                    "path": dir,
                    "files": ["*"],
                },
                "destination": {
                    "path": _TEMP_RESOURCES_DIR + "/"
                }
            }
            for dir in ios_res_directories
        ]
        return (ios_packager_temp_resources
                if ios_packager_temp_resources
                else [
                    {
                        "source": {
                            "type": "DIRECTORY"
                        },
                        "destination": {
                            "path": _TEMP_RESOURCES_DIR
                        }
                    }
                ])

    def _empty_dir_for_mangled_symbols_data(self):
        return [
                   {
                       "source": {
                           "type": "DIRECTORY"
                       },
                       "destination": {
                           "path": "/share/maps/mobile/bundle/ios"
                       }
                   }
               ]

    def _mangler_data(self):
        return [
                   {
                       "source": self._mangler_source(),
                       "destination": {
                           "path": "/bin/tools-mangle-symbols"
                       },
                   }
               ]

    def _mangler_source(self):
        xcode_toolchain_version = self._toolchains_config["ios"]["toolchain_prefix"]
        if xcode_toolchain_version == "xcode_12_5":
            resource_id = "2582237896"
        else:
            raise TaskError('Unknown xcode version {xcode_toolchain_version}.'.format(xcode_toolchain_version=xcode_toolchain_version))

        return {
                   "type": "SANDBOX_RESOURCE",
                   "id": resource_id,
                   "path": "tools-mangle-symbols",
                   "untar": True,
               }

    def _postprocess(self):
        postprocess = []
        if not self.Parameters.is_dynamic:
            if self.Parameters.mangle_symbols:
                postprocess += self._postprocess_mangle()

            postprocess += self._postprocess_lipo()

        postprocess += self._postprocess_ios_packager()
        if self.Parameters.is_navi:
            postprocess += self._postprocess_mv_local_headers()
        return postprocess

    def _postprocess_mangle(self):
        return [
                   {
                       "source": self._mangler_source(),
                       "arguments": [
                           './' + self._temp_library_basename(arch),
                           '--out-renamed', ('./share/maps/mobile/bundle/ios/mangled_symbols.{arch}'
                                             .format(arch=arch)),
                       ],
                   }
                   for arch in self._architectures
               ]

    def _postprocess_lipo(self):
        if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_M1_SIMULATOR_BUILD):
            return [
                {
                    "source": {
                        "type": "SANDBOX_RESOURCE",
                        "id": "1130399306",
                        "path": "bin/lipo",
                        "untar": True
                    },
                    "arguments": (["-create"]
                                  + [self._temp_library_basename(arch)
                                     for arch in ["x86_64", "m1sim"]]
                                  + ["-output", self._temp_library_basename(arch='sim_universal')])
                }
            ]

        if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_STATIC_XCFRAMEWORK):
            return []

        return [
                   {
                       "source": {
                           "type": "SANDBOX_RESOURCE",
                           "id": "1130399306",
                           "path": "bin/lipo",
                           "untar": True
                       },
                       "arguments": (["-create"]
                                     + [self._temp_library_basename(arch)
                                        for arch in self._architectures]
                                     + ["-output", self._temp_library_basename(arch=None)])
                   }
               ]

    def _postprocess_ios_packager(self):
        return [
                   {
                       "source": {
                           "type": "BUILD_OUTPUT",
                           "build_key": "build_tools",
                           "path": self._ios_packager_binary_path()
                       },
                       "arguments":
                           self._ios_packager_dynamic_arg()
                           + self._ios_packager_libraries_arg()
                           + [
                               "--headers-temp", "./ios_packager_temp_headers",
                               "--resources-temp", ("." + _TEMP_RESOURCES_DIR),
                               "--module-name", self.Parameters.framework_name,
                               "--framework-name", self.Parameters.framework_name,
                               "--framework-version", "0",
                               "--header-prefixes", "",
                               "--modulemap-exclude-prefixes", "YMA"
                           ]
                   }
               ]

    def _ios_packager_dynamic_arg(self):
        if (not is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_UNITED_IOS_PACKAGER) or
               is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_M1_SIMULATOR_BUILD)):
            return []

        if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_STATIC_XCFRAMEWORK):
            return ["--xcframework"]
        return ["--dynamic"] if self.Parameters.is_dynamic else ["--static"]

    def _ios_packager_libraries_arg(self):
        if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_M1_SIMULATOR_BUILD):
            return ["--libraries"] + ["./" + self._temp_library_basename(arch)
                                      for arch in ["arm64", "sim_universal"]]
        if self.Parameters.is_dynamic or is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_STATIC_XCFRAMEWORK):
            return ["--libraries"] + ["./" + self._temp_library_basename(arch)
                                      for arch in self._architectures]
        if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_UNITED_IOS_PACKAGER):
            return ["--libraries", "./" + self._temp_library_basename(arch=None)]
        return ["--library", "./" + self._temp_library_basename(arch=None)]

    def _make_framework_zip(self):
        tempdir = os.path.abspath('framework_artifacts')
        extract_resource(sdk2.Resource[self.Context.framework_artifacts], tempdir)
        if self.Parameters.is_dynamic or is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_STATIC_XCFRAMEWORK):
            tempdir_paths = [
                os.path.join(tempdir, '{}.bundle'.format(self.Parameters.framework_name)),
                os.path.join(tempdir, '{}.xcframework'.format(self.Parameters.framework_name)),
            ]
        else:
            tempdir_paths = [
                os.path.join(tempdir, '{}.framework'.format(self.Parameters.framework_name)),
            ]
        framework_zip_path = '{}-{}.framework.zip'.format(
                self.Parameters.framework_name, self.Parameters.release_version)
        zip_bytes = pack_into_zip_file(tempdir, tempdir_paths)
        self.Parameters.framework_zip = make_resource(
                self,
                MapsMobileFrameworkZip,
                description='Framework zip',
                path=framework_zip_path,
                content=zip_bytes,
                ttl=_FRAMEWORK_ZIP_TTL,
                )

    def _deb_pkg_json_data(self):
        deb_pkg_json_data = self._common_deb_pkg_json_data()
        if self.Parameters.is_dynamic or is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITH_STATIC_XCFRAMEWORK):
            deb_pkg_json_data += self._dynamic_deb_pkg_json_data()
        else:
            deb_pkg_json_data += self._static_deb_pkg_json_data()
        return deb_pkg_json_data

    def _common_deb_pkg_json_data(self):
        return [
            {
                "source": {
                    "type": "SANDBOX_RESOURCE",
                    "id": self.Context.framework_artifacts,
                    "untar": True,
                },
                "destination": {
                    "path": "/tmp/",
                    "temp": True,
                }
            },
            {
                "source": {
                    "type": "TEMP",
                    "path": "tmp/local",
                },
                "destination": {
                    "path": "/local",
                }
            },
        ]

    def _dynamic_deb_pkg_json_data(self):
        return [
            {
                "source": {
                    "type": "TEMP",
                    "path": "tmp/{}.bundle".format(self.Parameters.framework_name),
                },
                "destination": {
                    "path": "/bundle/{}.bundle".format(self.Parameters.framework_name),
                },
            },
            {
                "source": {
                    "type": "TEMP",
                    "path": "tmp/{}.xcframework".format(self.Parameters.framework_name),
                },
                "destination": {
                    "path": "/bundle/{}.xcframework".format(self.Parameters.framework_name),
                },
            },
        ]

    def _static_deb_pkg_json_data(self):
        return [
            {
                "source": {
                    "type": "TEMP",
                    "path": "tmp/{}.framework/Versions/A".format(self.Parameters.framework_name),
                },
                "destination": {
                    "path": "/bundle/{}.framework/Versions/A".format(self.Parameters.framework_name),
                },
            },
            {
                "source": {
                    "type": "TEMP",
                    "path": "tmp/{}.framework/Modules".format(self.Parameters.framework_name),
                },
                "destination": {
                    "path": "/bundle/{}.framework/Modules".format(self.Parameters.framework_name),
                },
            },
            {
                "source": {
                    "type": "SYMLINK",
                },
                "destination": {
                    "target": "A",
                    "path": ("/bundle/{}.framework/Versions/Current"
                             .format(self.Parameters.framework_name)),
                },
            },
            {
                "source": {
                    "type": "SYMLINK",
                },
                "destination": {
                    "target": "Versions/Current/{}".format(self.Parameters.framework_name),
                    "path": ("/bundle/{framework_name}.framework/{framework_name}"
                             .format(framework_name=self.Parameters.framework_name)),
                },
            },
            {
                "source": {
                    "type": "SYMLINK",
                },
                "destination": {
                    "target": "Versions/Current/Headers",
                    "path": "/bundle/{}.framework/Headers".format(self.Parameters.framework_name),
                },
            },
            {
                "source": {
                    "type": "SYMLINK",
                },
                "destination": {
                    "target": "Versions/Current/Resources",
                    "path": "/bundle/{}.framework/Resources".format(self.Parameters.framework_name),
                },
            },
            {
                "source": {
                    "type": "TEMP",
                    "path": "tmp/share",
                },
                "destination": {
                    "path": "/share",
                }
            },
        ]
