import os
import logging
import multiprocessing
import shutil
import urllib
import base64
import re
import yaml
import platform

from sandbox import sdk2
import sandbox.common.types.client as ctc
import sandbox.projects.common.arcadia.sdk as arcadia_sdk
from sandbox.projects.maps.common.retry import retry
from sandbox.projects.maps.mobile.MapsMobileBuildPlatformProject import MapsMobileBuildPlatformProject
from sandbox.projects.maps.mobile.utils.resource_helpers import extract_resource
from sandbox.projects.maps.mobile.utils.subprocess_helpers import check_call_wrapper
from sandbox.projects.maps.mobile.utils.arcadia_url_helpers import is_branch_up_to_date

_SIGNING_IDENTITY_MAP = {
    'development': 'iPhone Developer',
    'enterprise': 'iPhone Distribution: Yandex LLC'
}

_SERVER_ENVIRONMENT_MAP = {
    'release': 'production',
    'servertesting': 'testing',
    'debug': 'production'
}

_BIG_SUR = "10.16"
_XCODE_VERSION_FOR_BIG_SUR = "13"

_FIRST_BRANCH_WITH_NEW_IPA_TESTING_NAME = '2021011012'

_FIRST_BRANCH_WITH_STATIC_AND_DYNAMIC_SCHEMES = '20210211'  # to replace with actual branch name that is >= 20210211
_FIRST_TRUNK_COMMIT_WITH_STATIC_AND_DYNAMIC_SCHEMES = 7846866
_FIRST_BRANCH_WITHOUT_STATIC_AND_DYNAMIC_SCHEMES = '20220131'

_EMULATOR_ARCHS = {'i386', 'x86_64'}
_DEVICE_ARCHS = {'arm64', 'armv7'}

_KEYCHAIN_NAME = 'mobile-signing'
_CODESIGN_PATH = os.path.join(os.sep, 'usr', 'bin', 'codesign')

_PROVISIONING_PROFILE_URL_TEMPLATE = 'https://bitbucket.browser.yandex-team.ru/projects/MT/repos/mobile-ios-provisioning-profiles/raw/AdHoc/{}.mobileprovision'
_PROVISIONING_PROFILE_DIR = os.path.join(os.path.expanduser('~'), 'Library', 'MobileDevice', 'Provisioning Profiles')
_PROVISIONING_PROFILE_FILENAME_TEMPLATE = '{}.mobileprovision'

_PROVISIONING_TEMPLATE = '''\t<key>provisioningProfiles</key>
\t<dict>
\t\t<key>{app_id}</key>
\t\t<string>{provision}</string>
\t</dict>
'''

_PLIST_TEMPLATE = '''<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE plist PUBLIC "-//Apple//DTD PLIST 1.0//EN" "http://www.apple.com/DTDs/PropertyList-1.0.dtd">
<plist version="1.0">
<dict>
\t<key>method</key>
\t<string>{method}</string>
\t<key>teamID</key>
\t<string>{team_id}</string>
{provisioning}
\t<key>compileBitcode</key>
\t<false/>
</dict>
</plist>
'''

_USER_NAME = 'robot-mapkit-ci'
_SSH_SECRET_NAME = 'ssh_key'

_SCHEME_PREFIX_TO_LINKING_SUFFIX = {
    'Dynamic': '',
    'Static': '_static',
}


class ThrowingUrlOpener(urllib.FancyURLopener):
    def http_error_default(self, req, fp, code, msg, hdrs):
        urllib.URLopener.http_error_default(self, req, fp, code, msg, hdrs)


class MapsMobileBuildIosApp(MapsMobileBuildPlatformProject):
    ''' Task for building mobile apps '''

    _bundle_task_type = 'MAPS_MOBILE_YA_PACKAGE_DARWIN'

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.CUSTOM_MAPS_MOBILE_DARWIN & ctc.Tag.USER_MAPS_MOBILE_DARWIN_IOS_APP

    class Parameters(MapsMobileBuildPlatformProject.Parameters):
        with sdk2.parameters.Group('App build') as app_block:
            app = sdk2.parameters.String('Application name, e.g. TestApp', required=True)
            scheme = sdk2.parameters.String('Scheme, e.g. StaticTestApp', required=True)
            export_app_id = sdk2.parameters.String('Export app id', required=True)
            xcode_version = sdk2.parameters.String('Force Xcode version', required=False)

        with sdk2.parameters.Group('Signing') as signing_block:
            team_id = sdk2.parameters.String('Team id', required=True)
            with sdk2.parameters.String('Signing method', required=True) as signing_method:
                signing_method.values.enterprise = signing_method.Value('enterprise', default=True)
                signing_method.values.development = 'development'
            signing_provision = sdk2.parameters.String('iOS provision profile name', required=True)
            signing_certificate = sdk2.parameters.YavSecret(
                    'Yav secret with signing certificate',
                    required=True
                    )
            keychain_password = sdk2.parameters.YavSecret(
                    'Yav secret with maps mobile keychain password',
                    required=True
                    )

        with sdk2.parameters.Group('Crashlytics') as crashlytics_block:
            crashlytics_config = sdk2.parameters.YavSecret(
                    'Yav secret with crashlytics key and secret',
                    required=True
                    )

    def _set_up(self):
        self._branch = self._arcadia_url_to_branch(self.Parameters.arcadia_url)
        self._scheme = (
            self.Parameters.scheme
            if self._branch != 'trunk' and self._branch >= _FIRST_BRANCH_WITH_STATIC_AND_DYNAMIC_SCHEMES
            or self._branch == 'trunk'
            and int(self.Parameters.arcadia_revision) >= _FIRST_TRUNK_COMMIT_WITH_STATIC_AND_DYNAMIC_SCHEMES
            else self.Parameters.app
        )

    def _parse_build_config(self):
        path_to_build_config = os.path.join('maps', 'mobile', '.build_config.yaml')
        with arcadia_sdk.mount_arc_path(self.Parameters.arcadia_url,
                                        use_arc_instead_of_aapi=True) as arcadia:
            with open(os.path.join(arcadia, path_to_build_config)) as f:
                return yaml.safe_load(f)

    def _get_developer_env(self):
        developer_env = os.environ
        developer_env['DEVELOPER_DIR'] = os.path.join(
            '/Applications',
            'Xcode{}.app'.format(self.xcode_version),
            'Contents',
            'Developer'
            )

    def _run_xcode_build(self, xcode_args, app_src_path):
        xcode_env = self._get_developer_env()
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('xcode_build')) as pl:
            check_call_wrapper(
                ['xcodebuild'] + xcode_args,
                    cwd=app_src_path,
                    env=xcode_env,
                    stdout=pl.stdout,
                    stderr=pl.stdout
                    )

    def _change_signing_style(self, style, app_src_path):
        filename = os.path.join(app_src_path, '{}.xcodeproj'.format(self.Parameters.app), 'project.pbxproj')
        with open(filename, 'r+') as proj:
            content = proj.read()
            new_content = re.sub(
                'ProvisioningStyle = .*;',
                'ProvisioningStyle = ' + style + ';',
                content,
                flags=re.M
            )
            proj.truncate(0)
            proj.seek(0)
            proj.write(new_content)

    def _clean_workspace(self, app_src_path):
        self._run_xcode_build([
            'clean',
            '-workspace',
            os.path.join(app_src_path, '{}.xcworkspace'.format(self.Parameters.app)),
            '-scheme',
            self._scheme
        ], app_src_path)

    def _default_xcode_args(self, app_src_path):
        env_vars = [
            'ONLY_ACTIVE_ARCH=NO',
            'BUILD_YANDEXMAPSMOBILEBUNDLE=NO',
            'DEBUG_INFORMATION_FORMAT=dwarf-with-dsym',
            'CRASHLYTICS_API_KEY=' + self.Parameters.crashlytics_config.data()['key'],
            'CRASHLYTICS_API_SECRET=' + self.Parameters.crashlytics_config.data()['secret'],
        ]
        if is_branch_up_to_date(self.Parameters.arcadia_url, self._FIRST_BRANCH_WITH_ICON_VERSION_WRITER):
            env_vars.append('ICON_VERSION_WRITER_PATH=' + os.path.dirname(self._icon_version_writer_path))
        return env_vars + [
            '-configuration', _SERVER_ENVIRONMENT_MAP[self.Parameters.build_variant],
            '-scheme', self._scheme,
            '-workspace',
            os.path.join(app_src_path, '{}.xcworkspace'.format(self.Parameters.app)),
            '-parallelizeTargets', '-jobs', str(multiprocessing.cpu_count())
        ]

    def _export_options_plist(self):
        app_id = self.Parameters.export_app_id
        plist = os.path.join(self._build_dir, 'export.plist')
        if self._signing_config['provision'] == '':
            provisioning_text = ''
        else:
            provisioning_text = _PROVISIONING_TEMPLATE.format(
                app_id=app_id,
                provision=self._signing_config['provision']
                )
        plist_text = _PLIST_TEMPLATE.format(
            method=self._signing_config['method'],
            provisioning=provisioning_text,
            team_id=self._signing_config['team-id']
            )
        with open(plist, 'w') as f:
            f.write(plist_text)
        return plist

    def _extract_framework(self):
        if os.path.isdir(self._bundle_dir):
            shutil.rmtree(self._bundle_dir)
        extract_resource(self.Parameters.output_bundle_resource, self._bundle_dir)

    def _build_for_device(self, app_src_path):
        self._extract_framework()

        default_args = self._default_xcode_args(app_src_path)
        device_xcode_args = default_args + [
            'archive',
            'ARCHS=' + ' '.join(_DEVICE_ARCHS & set(self.archs)),
            '-sdk', 'iphoneos',
            '-archivePath', self._build_dir + '/device'
        ]
        device_xcode_args.extend(['USE_CRASHLYTICS=1'])
        if self._signing_config['method'] == 'enterprise':
            self._change_signing_style('Manual', app_src_path)
            signingIdentity = _SIGNING_IDENTITY_MAP[self._signing_config['method']]
            try:
                self._run_xcode_build(
                    device_xcode_args
                    + [
                        'PROVISIONING_PROFILE_SPECIFIER='
                            + self._signing_config['team-id']
                            + '/'
                            + self._signing_config['provision'],
                        'CODE_SIGN_IDENTITY='
                            + signingIdentity,
                        'CODE_SIGN_STYLE=Manual',
                        'DEVELOPMENT_TEAM='
                            + self._signing_config['team-id']
                    ],
                    app_src_path
                )
            finally:
                self._change_signing_style('Automatic', app_src_path)
        else:
            self._run_xcode_build(device_xcode_args, app_src_path)
        plist = self._export_options_plist()
        self._run_xcode_build(['-exportArchive',
                        '-archivePath', os.path.join(self._build_dir, 'device.xcarchive'),
                        '-exportOptionsPlist', plist,
                        '-exportPath', os.path.join(self._build_dir, 'ipa')], app_src_path)

    def _build_for_emulator(self, app_src_path):
        self._extract_framework()
        default_args = self._default_xcode_args(app_src_path)
        emulator_xcode_args = default_args + [
            'build',
            'ARCHS=' + ' '.join(_EMULATOR_ARCHS & set(self.archs)),
            'VALID_ARCHS=i386 x86_64 armv7 arm64',
            '-sdk', 'iphonesimulator',
            '-archivePath',
            self._build_dir + '/emulator',
            'OBJROOT='
                + self._build_dir + '/objroot',
            'SYMROOT='
                + self._build_dir + '/symroot'
        ]
        self._run_xcode_build(emulator_xcode_args, app_src_path)

    def _run_security(self, args):
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('security')) as pl:
            check_call_wrapper(
                ['security'] + args,
                stdout=pl.stdout,
                stderr=pl.stdout
                )

    def _set_up_keychain(self):
        keychain_password = self.Parameters.keychain_password.data()['password']
        certificate_file = os.path.abspath('certificate.p12')
        with open(certificate_file, 'wb') as f:
            f.write(base64.b64decode(self.Parameters.signing_certificate.data()['certificate-p12']))

        self._run_security(['create-keychain', '-p', keychain_password, _KEYCHAIN_NAME])
        self._run_security(['unlock-keychain', '-p', keychain_password, _KEYCHAIN_NAME])
        self._run_security([
            'import', certificate_file,
            '-k', _KEYCHAIN_NAME,
            '-P', self.Parameters.signing_certificate.data()['certificate-password'],
            '-T', _CODESIGN_PATH])
        self._run_security(['set-key-partition-list', '-S', 'apple-tool:,apple:', '-k', keychain_password, _KEYCHAIN_NAME])
        self._run_security(['set-keychain-settings', '-lu', '-t', '3600', _KEYCHAIN_NAME])
        self._run_security(['list-keychains', '-s', _KEYCHAIN_NAME])

    @retry(tries=3, delay=10, backoff=2)
    def _install_pods(self, app_src_path):
        with sdk2.ssh.Key(self, _USER_NAME, _SSH_SECRET_NAME):
            pod_env = self._get_developer_env()
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('pod_install')) as pl:
                check_call_wrapper(
                    ['pod', 'install'],
                    cwd=app_src_path,
                    env=pod_env,
                    stdout=pl.stdout,
                    stderr=pl.stdout
                    )

    @retry(tries=3, delay=10, backoff=2)
    def _retrieve_provisioning_profile(self):
        if not os.path.exists(_PROVISIONING_PROFILE_DIR):
            os.makedirs(_PROVISIONING_PROFILE_DIR)

        ThrowingUrlOpener().retrieve(
            _PROVISIONING_PROFILE_URL_TEMPLATE.format(self.Parameters.signing_provision),
            os.path.join(_PROVISIONING_PROFILE_DIR, _PROVISIONING_PROFILE_FILENAME_TEMPLATE.format(self.Parameters.signing_provision))
            )

    def _set_up_structure(self, app_src_path):
        self._build_dir = os.path.abspath('build')
        self._bundle_dir = os.path.join(app_src_path, 'bundle')

        self._retrieve_provisioning_profile()

        self._set_up_keychain()

    def _arcadia_url_to_branch(self, url):
        return (
            'trunk'
            if url.endswith('/trunk/arcadia')
            else re.sub(r'^.*maps-mobile-releases/(\d*)/arcadia', r'\1', url)
        )

    def _get_ipa_file_name(self):
        if (self.Parameters.build_variant != 'servertesting'
                or self._branch < _FIRST_BRANCH_WITH_NEW_IPA_TESTING_NAME):
            return '{}.ipa'.format(self.Parameters.app)
        return '{}.ipa'.format(self.Parameters.app + '.' + _SERVER_ENVIRONMENT_MAP[self.Parameters.build_variant])

    def _get_project_path(self, app_src_path):
        archs = set(self.archs)
        if len(_DEVICE_ARCHS & archs) != 0:
            ipa_file = self._get_ipa_file_name()
            ipa_dir = os.path.join(self._build_dir, 'ipa')
            ipa_path = os.path.join(ipa_dir, ipa_file)
            return ipa_path
        return None

    def _get_xcode_version(self, build_config):
        if self.Parameters.xcode_version:
            return self.Parameters.xcode_version
        mac_version = platform.mac_ver()[0]
        return _XCODE_VERSION_FOR_BIG_SUR if mac_version >= _BIG_SUR else build_config["ios"]["xcode_version"]

    def _run_build(self, app_src_path):
        self._set_up()
        build_config = self._parse_build_config()
        self.xcode_version = self._get_xcode_version(build_config)
        logging.info("Xcode version %s is used", self.xcode_version)

        self.archs = set(build_config["ios"]["architecture_to_local_dir"].keys())

        self._install_pods(app_src_path)
        self._clean_workspace(app_src_path)

        if len(_DEVICE_ARCHS & self.archs) != 0:
            self._signing_config = {
                'team-id': self.Parameters.team_id,
                'method': self.Parameters.signing_method,
                'provision': self.Parameters.signing_provision.replace('_', ' ')
            }
            self._build_for_device(app_src_path)

        if len(_EMULATOR_ARCHS & self.archs) != 0:
            self._build_for_emulator(app_src_path)

    def _beta_branch(self):
        branch = super(MapsMobileBuildIosApp, self)._beta_branch()
        if is_branch_up_to_date(self.Parameters.arcadia_url, _FIRST_BRANCH_WITHOUT_STATIC_AND_DYNAMIC_SCHEMES):
            return branch

        linking_suffix = _SCHEME_PREFIX_TO_LINKING_SUFFIX.get(next(
            (
                scheme_prefix for scheme_prefix in _SCHEME_PREFIX_TO_LINKING_SUFFIX
                if self.Parameters.scheme and self.Parameters.scheme.startswith(scheme_prefix)
            ), ''
        ), '')
        return '{}{}'.format(branch, linking_suffix)

    def on_execute(self):
        try:
            MapsMobileBuildPlatformProject.on_execute(self)
        finally:
            logging.info('Setting ttl 7 on log resource {}.'.format(self.log_resource.id))
            self.log_resource.ttl = 7
