import os
import logging
import shutil

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.maps.mobile.MapsMobileBuildIosProject import MapsMobileBuildIosProject
from sandbox.projects.maps.mobile.MapsMobileResources import MapsMobileIosFramework
from sandbox.projects.maps.mobile.MapsMobileResources import MapsMobileIosResourceBundle
from sandbox.projects.maps.mobile.utils.subprocess_helpers import check_call_wrapper


class MapsMobileBuildIosFatProject(MapsMobileBuildIosProject):
    ''' Task for building ios fat frameworks and bundles '''

    class Parameters(MapsMobileBuildIosProject.Parameters):
        with sdk2.parameters.Group('App build') as project_block:
            project_type = sdk2.parameters.String('Project type, e.g. Framework, Bundle, etc.', required = True)

    def _create_resource(self, project_file, revision):
        if self.Parameters.project_type == 'Bundle':
            return MapsMobileIosResourceBundle(
                self,
                '{} version {}'.format(project_file, revision),
                '{}.tgz'.format(project_file),
                ttl=31,
                backup_task=True
                )
        elif self.Parameters.project_type == 'Framework':
            return MapsMobileIosFramework(
                self,
                '{} version {}'.format(project_file, revision),
                '{}.tgz'.format(project_file),
                ttl=31,
                backup_task=True
                )
        else:
            raise TaskError('Unsupported project type')

    def _get_project_path(self, project_src_path):
        return self._result_product_path

    def _merge_into_fat(self, device_product_path, emulator_product_path):
        shutil.copytree(device_product_path, self._result_product_path, symlinks=True)
        os.remove(os.path.join(self._result_product_path, self.Parameters.project))

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('lipo')) as pl:
            check_call_wrapper(
                [
                    'lipo',
                    '-create',
                    os.path.join(device_product_path, self.Parameters.project),
                    os.path.join(emulator_product_path, self.Parameters.project),
                    '-output',
                    os.path.join(self._result_product_path, self.Parameters.project)
                ],
                stdout=pl.stdout,
                stderr=pl.stdout
                )

        if os.path.exists(os.path.join(emulator_product_path, 'Modules')):
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('rsync')) as pl:
                check_call_wrapper(
                    [
                        'rsync',
                        '-a',
                        os.path.join(emulator_product_path, 'Modules', self.Parameters.project + '.swiftmodule/'),
                        os.path.join(self._result_product_path, 'Modules', self.Parameters.project + '.swiftmodule/')
                    ],
                    stdout=pl.stdout,
                    stderr=pl.stdout
                    )

    def _run_build(self, project_src_path):
        MapsMobileBuildIosProject._run_build(self, project_src_path)

        product_path = os.path.join(
            'Products',
            'Library',
            self.Parameters.project_type + 's',
            self.Parameters.project + '.' + self.Parameters.project_type.lower())

        device_product_path = os.path.join(self._build_dir, 'device.xcarchive', product_path)
        emulator_product_path = os.path.join(self._build_dir, 'emulator.xcarchive', product_path)
        self._result_product_path = os.path.join(self._result_dir, self.Parameters.project + '.' + self.Parameters.project_type.lower())

        if not self._has_emulator_archs:
            shutil.copytree(device_product_path, self._result_product_path, symlinks=True)
        elif not self._has_device_archs:
            shutil.copytree(emulator_product_path, self._result_product_path, symlinks=True)
        else:
            self._merge_into_fat(device_product_path, emulator_product_path)
