import os
import logging
import multiprocessing
import shutil
import yaml

from sandbox import sdk2
import sandbox.common.types.client as ctc
import sandbox.projects.common.arcadia.sdk as arcadia_sdk
from sandbox.projects.maps.mobile.MapsMobileBuildPlatformProject import MapsMobileBuildPlatformProject
from sandbox.projects.maps.mobile.utils.resource_helpers import extract_resource
from sandbox.projects.maps.mobile.utils.subprocess_helpers import check_call_wrapper


_EMULATOR_ARCHS = {'x86_64'}
_DEVICE_ARCHS = {'arm64'}

class MapsMobileBuildIosProject(MapsMobileBuildPlatformProject):
    ''' Task for building iOs projects '''

    _bundle_task_type = 'MAPS_MOBILE_YA_PACKAGE_DARWIN'

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.CUSTOM_MAPS_MOBILE_DARWIN & ctc.Tag.USER_MAPS_MOBILE_DARWIN_IOS_APP

    class Parameters(MapsMobileBuildPlatformProject.Parameters):
        with sdk2.parameters.Group('App build') as project_block:
            project = sdk2.parameters.String('Application name, e.g. TestApp', required=True)
            scheme = sdk2.parameters.String('Scheme, e.g. StaticTestApp', required=True)

    def _parse_build_config(self):
        path_to_build_config = os.path.join('maps', 'mobile', '.build_config.yaml')
        with arcadia_sdk.mount_arc_path(self.Parameters.arcadia_url,
                                        use_arc_instead_of_aapi=True) as arcadia:
            with open(os.path.join(arcadia, path_to_build_config)) as f:
                return yaml.safe_load(f)

    def _get_developer_env(self):
        developer_env = os.environ
        developer_env['DEVELOPER_DIR'] = os.path.join(
            '/Applications',
            'Xcode{}.app'.format(self.xcode_version),
            'Contents',
            'Developer'
            )

    def _run_xcode_build(self, xcode_args, project_src_path):
        xcode_env = self._get_developer_env()
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('xcode_build')) as pl:
            check_call_wrapper(
                ['xcodebuild'] + xcode_args,
                cwd=project_src_path,
                env=xcode_env,
                stdout=pl.stdout,
                stderr=pl.stdout
                )

    def _default_xcode_args(self, project_src_path):
        env_vars = [
            'ONLY_ACTIVE_ARCH=NO',
            'BUILD_YANDEXMAPSMOBILEBUNDLE=NO',
            'SKIP_INSTALL=NO',
            'DEBUG_INFORMATION_FORMAT=dwarf-with-dsym',
            'YANDEXMAPSMOBILE_DIR='+self._bundle_dir
        ]
        return env_vars + [
            '-scheme', self.Parameters.scheme,
            '-parallelizeTargets', '-jobs', str(multiprocessing.cpu_count())
        ]

    def _extract_framework(self):
        if os.path.isdir(self._bundle_dir):
            shutil.rmtree(self._bundle_dir)
        extract_resource(self.Parameters.output_bundle_resource, self._bundle_dir)

    def _build_for_device(self, project_src_path):
        self._extract_framework()

        default_args = self._default_xcode_args(project_src_path)
        device_xcode_args = default_args + [
            'archive',
            'ARCHS=' + ' '.join(_DEVICE_ARCHS & set(self.archs)),
            '-sdk', 'iphoneos',
            '-archivePath', self._build_dir + '/device'
        ]
        self._run_xcode_build(device_xcode_args, project_src_path)

    def _build_for_emulator(self, project_src_path):
        self._extract_framework()
        default_args = self._default_xcode_args(project_src_path)
        emulator_xcode_args = default_args + [
            'archive',
            'ARCHS=' + ' '.join(_EMULATOR_ARCHS & set(self.archs)),
            'VALID_ARCHS=i386 x86_64 armv7 arm64',
            '-sdk', 'iphonesimulator',
            '-archivePath',
            self._build_dir + '/emulator'
        ]
        self._run_xcode_build(emulator_xcode_args, project_src_path)

    def _set_up_structure(self, project_src_path):
        self._build_dir = os.path.abspath('build')
        self._bundle_dir = os.path.join(project_src_path, 'bundle')
        self._result_dir = os.path.join(self._build_dir, 'result')

    def _run_build(self, project_src_path):
        build_config = self._parse_build_config()
        self.xcode_version = build_config["ios"]["xcode_version"]
        self.archs = set(build_config["ios"]["architecture_to_local_dir"].keys())
        self._has_device_archs = len(_DEVICE_ARCHS & self.archs) != 0
        self._has_emulator_archs = len(_EMULATOR_ARCHS & self.archs) != 0

        if self._has_device_archs:
            self._build_for_device(project_src_path)

        if self._has_emulator_archs:
            self._build_for_emulator(project_src_path)

    def on_execute(self):
        try:
            MapsMobileBuildPlatformProject.on_execute(self)
        finally:
            logging.info('Setting ttl 7 on log resource {}.'.format(self.log_resource.id))
            self.log_resource.ttl = 7
