import os
import tarfile
import tempfile

from sandbox import sdk2
from sandbox.sandboxsdk import svn
from sandbox.common.errors import TaskError
import sandbox.common.types.task as task_types
import sandbox.projects.common.arcadia.sdk as arcadia_sdk
import sandbox.projects.common.constants as consts
from sandbox.projects.maps.mobile.MapsMobileResources import MapsMobileApp

from sandbox.projects.maps.mobile.utils.resource_helpers import extract_resource
from sandbox.projects.maps.mobile.utils.subprocess_helpers import process_subtask_status
from sandbox.projects.maps.mobile.utils.subtask_runner import SubtaskRunner
from sandbox.projects.maps.mobile.utils.arcadia_url_helpers import is_branch_up_to_date


_BETA_BRANCH_TESTING_SUFFIX_MAP = {
    'release': '',
    'servertesting': '_testing'
}


class MapsMobileBuildPlatformProject(sdk2.Task):
    ''' Task for building mobile projects '''

    _FIRST_BRANCH_WITH_ICON_VERSION_WRITER = '20210411'  # todo replace with actual branch

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Repository'):
            arcadia_url = sdk2.parameters.ArcadiaUrl('Svn url for arcadia', required=True)
            use_svn = sdk2.parameters.Bool('Use SVN', default=True, required=False)
            with use_svn.value[True]:
                arcadia_revision = sdk2.parameters.String('Arcadia revision', required=True)
            arcadia_patch = sdk2.parameters.String('Apply patch', required=False)

        with sdk2.parameters.Group('Bundle build') as bundle_block:
            prebuilt_bundle = sdk2.parameters.Bool('Use prebuilt bundle', default=False, required=True)
            with prebuilt_bundle.value[True]:
                bundle_resource = sdk2.parameters.Resource('Bundle resource', required=True)

            with prebuilt_bundle.value[False]:
                package_config = sdk2.parameters.String(
                    'Bundle package config, relative to Arcadia',
                    required=True
                    )

        with sdk2.parameters.Group('Project build') as project_block:
            project_src_path = sdk2.parameters.String('Project src path relative to Arcadia')
            # TODO: remove this parameter. It is here for branches backward compatibility.
            app_src_path = sdk2.parameters.String('App src path relative to Arcadia')
            build_variant = sdk2.parameters.String('Build variant', required=True)

        with sdk2.parameters.Output:
            artifact_resource = sdk2.parameters.Resource('Artifact resource')
            artifact_path = sdk2.parameters.String('Artifact path inside resource')
            beta_branch = sdk2.parameters.String('Beta branch for application')
            output_bundle_resource = sdk2.parameters.Resource('Bundle resource')

    def _find_task(self, task_id):
        for child_task in self.find():
            if child_task.id == task_id:
                return child_task
        raise TaskError('Did not find YA_PACKAGE child task')

    def _build_bundle(self):
        with self.memoize_stage.build_bundle:
            package_args = self._common_ya_package_parameters()
            package_args.update({
                'packages': self.Parameters.package_config,
                'ya_timeout': 6 * 60 * 60
            })
            task_class = sdk2.Task[self._bundle_task_type]
            ya_package_task = task_class(
                self,
                description='YA_PACKAGE for bundle building. Subtask of {}'.format(self.id),
                inherit_notifications=True,
                **package_args
                )
            ya_package_task.enqueue()
            self.Context.ya_package_task_id = ya_package_task.id

            raise sdk2.WaitTask(
                ya_package_task,
                [task_types.Status.Group.FINISH, task_types.Status.Group.BREAK], wait_all=True
                )

        ya_package_task = self._find_task(self.Context.ya_package_task_id)
        process_subtask_status(ya_package_task.status, 'Could not build bundle. Subtask {} failed.'.format(ya_package_task.id))
        resource = sdk2.Resource['YA_PACKAGE'].find(task=ya_package_task).first()
        if not resource:
            raise TaskError('Could not find bundle resource')
        with self.memoize_stage.get_resource:
            self.Parameters.output_bundle_resource = resource

    def _build_icon_version_writer(self):
        parameters = self._common_ya_package_parameters()
        parameters.update({
            'packages': 'maps/mobile/tools/icon_version_writer/package/pkg.json',
        })
        tag = 'build_icon_version_writer'
        SubtaskRunner(self).run_and_handle_result(self._bundle_task_type, parameters, tag=tag,
                                                  result_callback=self._build_icon_version_writer_callback)

    def _build_icon_version_writer_callback(self, subtask):
        resource = sdk2.Resource['YA_PACKAGE'].find(task=subtask).first()
        self._icon_version_writer_path = os.path.abspath(
                os.path.join('build', 'icon_version_writer', 'icon_version_writer'))
        extract_resource(resource, os.path.dirname(self._icon_version_writer_path))

    def _common_ya_package_parameters(self):
        return {
                consts.ARCADIA_URL_KEY: self._create_arcadia_url(),
                consts.ARCADIA_PATCH_KEY: self.Parameters.arcadia_patch,
                consts.USE_AAPI_FUSE: True,
                consts.USE_ARC_INSTEAD_OF_AAPI: True,
                'package_type': 'tarball',
                'build_system': 'semi_distbuild',
                'force_vcs_info_update': True,
        }

    def _checkout_arcadia_src(self):
        return arcadia_sdk.mount_arc_path(self._create_arcadia_url(),
                                          use_arc_instead_of_aapi=True)

    def _create_arcadia_url(self):
        if self.Parameters.use_svn:
            return '{}@{}'.format(self.Parameters.arcadia_url,
                                  self.Parameters.arcadia_revision)
        return self.Parameters.arcadia_url

    def _set_up_structure(self, project_src_path):
        raise NotImplementedError()

    def _get_project_path(self, project_src_path):
        raise NotImplementedError()

    def _get_resource_files_paths(self, project_src_path):
        return [self._get_project_path(project_src_path)]

    def _run_build(self, project_src_path):
        raise NotImplementedError()

    def _create_resource(self, project_file, revision):
        return MapsMobileApp(
            self,
            '{} version {}'.format(project_file, revision),
            '{}.tgz'.format(project_file),
            ttl=31,
            backup_task=True
            )

    def _output_resource(self, project_src_path):
        if self.Parameters.use_svn:
            revision = self.Parameters.arcadia_revision
        else:
            revision = self.Parameters.arcadia_url.split('#')[-1]

        project_path = self._get_project_path(project_src_path)
        project_file = os.path.basename(project_path)
        resource = self._create_resource(project_file, revision)
        resource.version = revision
        resource_data = sdk2.ResourceData(resource)
        resource_paths = self._get_resource_files_paths(project_src_path)
        with tarfile.open(str(resource_data.path), 'w:gz') as tar:
            for resource_path in resource_paths:
                resource_file = os.path.basename(resource_path)
                tar.add(resource_path, arcname=resource_file)
        self.Parameters.artifact_path = resource_file
        resource_data.ready()
        return resource, project_file

    def _build_project(self):
        with self.memoize_stage.build_project(commit_on_entrance=False):
            with self._checkout_arcadia_src() as arcadia_path:
                project_src_path = os.path.join(arcadia_path, self.project_src_path)
                if self.Parameters.arcadia_patch:
                    # We use apply_patch from svn sdk because arc sdk doesn't have a similar method.
                    # It does not seem to run svn subprocesses anyway.
                    svn.Arcadia.apply_patch(arcadia_path, self.Parameters.arcadia_patch, tempfile.mkdtemp())
                self._set_up_structure(project_src_path)
                self._run_build(project_src_path)
                if self._get_project_path(project_src_path):
                    self.Parameters.artifact_resource, self._artifact_path = self._output_resource(project_src_path)

    def _beta_branch(self):
        testing_suffix = _BETA_BRANCH_TESTING_SUFFIX_MAP[self.Parameters.build_variant]
        url_tail = self.Parameters.arcadia_url.split('/')[-2]
        branch = url_tail if url_tail == 'trunk' else 'r{}'.format(url_tail)
        return '{}{}'.format(branch, testing_suffix)

    def on_execute(self):
        self.project_src_path = self.Parameters.project_src_path or self.Parameters.app_src_path
        if self.Parameters.prebuilt_bundle:
            with self.memoize_stage.set_output_bundle:
                self.Parameters.output_bundle_resource = self.Parameters.bundle_resource
        else:
            self._build_bundle()
        if is_branch_up_to_date(self.Parameters.arcadia_url, self._FIRST_BRANCH_WITH_ICON_VERSION_WRITER):
            self._build_icon_version_writer()
        self._build_project()

        beta_branch = self._beta_branch()
        if beta_branch in ('trunk', 'trunk_static', 'trunk_testing', 'trunk_testing_static'):
            beta_branch += '_testenv'
        self.Parameters.beta_branch = beta_branch
