import json
import re

from sandbox import sdk2
from sandbox.projects.maps.mobile.utils.subtask_runner import SubtaskRunner, TaskError


def _get_branch(arcadia_url):
    match = re.match(r'^.*maps-mobile-releases/(\d*)/arcadia', arcadia_url)
    if match:
        return match.group(1)
    return None


_BRANCH_REVISION = '{branch}.{revision}'


class MapsMobileBumpMapkitInNavikit(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):

        build_navi_release = sdk2.parameters.Bool('Build and export new navi release')

        with build_navi_release.value[True]:
            arcadia_url = sdk2.parameters.ArcadiaUrl('Svn url for arcadia')
            arcadia_revision = sdk2.parameters.String('Arcadia revision')

        with build_navi_release.value[False]:
            mapkit_version = sdk2.parameters.String('Mapkit release version. Example: 2021111818.8893839')

        ssh_key = sdk2.parameters.String(
            'SSH key',
            description='SSH key for TEAMCITY_SANDBOX_RUNNER_LAUNCHER',
            required=True)

        config_path = sdk2.parameters.String(
            'Config path',
            description='Config path for TEAMCITY_SANDBOX_RUNNER_LAUNCHER',
            default='mobile/geo/maps/maps/tools/bump-mapkit/bump-mapkit.',
            required=True)

        dry_run = sdk2.parameters.Bool('Dry run (add "RUN_MODE": "--dry" argument to TSR)', default=False)

    def on_execute(self):
        subtask_runner = SubtaskRunner(self)

        if self.Parameters.build_navi_release:
            branch = _get_branch(self.Parameters.arcadia_url)
            if not branch:
                raise TaskError('Cannot get svn branch from arcadia url: {}'.format(self.Parameters.arcadia_url))

            branch_revision = _BRANCH_REVISION.format(branch=branch, revision=self.Parameters.arcadia_revision)

            with self.memoize_stage.export_mapkit:
                subtask_runner.run(
                    'MAPS_MOBILE_EXPORT_ALL',
                    parameters={
                        'branch_revision': branch_revision,
                        'classifiers': ['navi', 'navinomangle']
                    },
                    wait=True
                )

            SubtaskRunner.require_all_successful(subtask_runner.find_tasks())
        else:
            branch_revision = self.Parameters.mapkit_version

        with self.memoize_stage.run_tsr:
            tsr_env = {
                'MAPKIT_VERSION': branch_revision
            }
            if self.Parameters.dry_run:
                tsr_env['RUN_MODE'] = '--dry'

            subtask_runner.run(
                'TEAMCITY_SANDBOX_RUNNER_LAUNCHER',
                parameters={
                    'release_type': 'stable',
                    'repo_url': 'arcadia',
                    'config_from_repository': True,
                    'ssh_key': self.Parameters.ssh_key,
                    'config_path': self.Parameters.config_path,
                    'env': json.dumps(tsr_env)
                },
                wait=True)
        SubtaskRunner.require_all_successful(subtask_runner.find_tasks())
