import re
import sys

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.maps.mobile.utils.subtask_runner import SubtaskRunner


_BRANCH_REVISION_REGEX = re.compile('^(\d+)\.(\d+)$')

_SIMPLE_EXPORT_SUBTASK_TYPE = 'MAPS_MOBILE_EXPORT_ARTIFACTS'
_NAVI_DEV_BUNDLE_EXPORT_SUBTASK_TYPE = 'MAPS_MOBILE_EXPORT_DEV_BUNDLE'
_NAVI_PLATFORM_EXPORT_SUBTASK_TYPE = 'MAPS_MOBILE_EXPORT_NAVI_ARTIFACTS'


_COMMON_PLATFORMS = [
    'android',
    'ios',
]

_NAVI_PLATFORMS = _COMMON_PLATFORMS + [
    'linux',
    'darwin',
]

_SIMPLE_BUNDLE_CLASSIFIER_TO_LIBRARIES = {
    'lite': [
        'mapkit',
    ],
    'edadeal': [
        'mapkit',
        'datasync',
    ],
    'full': [
        'mapkit',
        'search',
        'directions',
        'places',
        'transport',
    ],
    'drive': [
        'mapkit',
        'directions',
        'search'
    ],
    'realty': [
        'mapkit',
        'places',
    ],
    'search': [
        'mapkit',
        'transport',
        'datasync',
    ],
    'zapravki': [
        'mapkit',
        'directions',
        'search'
    ],
    'taxi': [
        'mapkit',
        'transport',
        'directions'
    ],
    'mrc': [
        'mapkit',
        'directions',
        'mrc',
        'auth',
        'recording',
        'places'
    ],
    'autoru': [
        'mapkit',
        'places',
        'search'
    ],
    'datasync': [
        'datasync'
    ]
}

_SIMPLE_BUNDLE_CLASSIFIER_TO_PLATFORM_LIBRARIES = {
    'full': [
        'styling_car_navigation',
        'styling_road_events',
        'styling_transport_navigation',
    ],
    'taxi': [
        'styling_car_navigation',
        'styling_road_events',
        'styling_transport_navigation',
    ]
}

_NAVI_LIBRARIES = [
    'mapkit',
    'search',
    'directions',
    'places',
    'transport',
    'bookmarks',
    'datasync',
    'push',
    'recording',
    'mrc'
]


class NaviBundle:
    def __init__(self, libraries, platforms, first_branch=None, ios_build_dynamic=False, ios_mangle_symbols=True):
        self.libraries = libraries
        self.platforms = platforms
        self.first_branch = first_branch
        self.ios_build_dynamic = ios_build_dynamic
        self.ios_mangle_symbols = ios_mangle_symbols


_NAVI_BUNDLES = {
    'navi': NaviBundle(_NAVI_LIBRARIES, _NAVI_PLATFORMS, ios_mangle_symbols=False),
    'navinomangle': NaviBundle(_NAVI_LIBRARIES, ['ios'], ios_mangle_symbols=False)
}


class MapsMobileExportAll(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):

        branch_revision = sdk2.parameters.String(
                '<branch>.<revision>:',
                description='An example: 2020050617.6789000',
                required=True)

        with sdk2.parameters.CheckGroup('Classifiers') as classifiers:
            for cl in _SIMPLE_BUNDLE_CLASSIFIER_TO_LIBRARIES:
                classifiers.values[cl] = classifiers.Value(cl, checked=True)
            for cl in _NAVI_BUNDLES:
                classifiers.values[cl] = classifiers.Value(cl, checked=False)

    def on_execute(self):
        with self.memoize_stage.run:
            self._set_up()
            for classifier in _SIMPLE_BUNDLE_CLASSIFIER_TO_LIBRARIES:
                if classifier in self.Parameters.classifiers:
                    for platform in _COMMON_PLATFORMS:
                        self._export_simple(classifier, platform)
            for classifier, navi_bundle in _NAVI_BUNDLES.items():
                if classifier in self.Parameters.classifiers:
                    if navi_bundle.first_branch is not None \
                        and self.Parameters.branch_revision < navi_bundle.first_branch:
                        print >> sys.stderr, "Configuration %s skipped due to minimal source version" % classifier

                    self._export_dev_bundle(classifier)
                    for platform in navi_bundle.platforms:
                        self._export_navi_platform(classifier, platform)
            subtasks = SubtaskRunner(self).find_tasks()
            SubtaskRunner.wait_all(subtasks)
        with self.memoize_stage.check:
            subtasks = SubtaskRunner(self).find_tasks()
            SubtaskRunner.require_all_successful(subtasks)

    def _set_up(self):
        self._make_arcadia_url()

    def _make_arcadia_url(self):
        match = _BRANCH_REVISION_REGEX.match(self.Parameters.branch_revision)
        if not match:
            self._raise_invalid_branch_revision()
        branch = match.group(1)
        revision = match.group(2)
        self.Context.arcadia_url = 'arcadia:/arc/branches/maps-mobile-releases/{}/arcadia@{}'.format(branch, revision)

    def _raise_invalid_branch_revision(self):
        raise TaskFailure('<branch>.<revision> does not match the pattern of {}'.format(_BRANCH_REVISION_REGEX.pattern))

    def _export_simple(self, classifier, platform):
        subtask_type = _SIMPLE_EXPORT_SUBTASK_TYPE
        libraries = _SIMPLE_BUNDLE_CLASSIFIER_TO_LIBRARIES[classifier]
        platform_libraries = _SIMPLE_BUNDLE_CLASSIFIER_TO_PLATFORM_LIBRARIES.get(classifier, None)
        parameters = self._common_parameters(classifier, libraries, platform, platform_libraries = platform_libraries,
                                             ios_dynamic=False, ios_mangle_symbols=True)
        parameters.update(self._simple_parameters())
        self._export(subtask_type, parameters, classifier)

    def _export_dev_bundle(self, classifier):
        subtask_type = _NAVI_DEV_BUNDLE_EXPORT_SUBTASK_TYPE
        navi_bundle = _NAVI_BUNDLES[classifier]
        libraries = navi_bundle.libraries
        parameters = self._common_parameters(classifier, libraries, platform=None)
        parameters.update(self._navi_parameters())
        self._export(subtask_type, parameters, classifier)

    def _export_navi_platform(self, classifier, platform):
        subtask_type = _NAVI_PLATFORM_EXPORT_SUBTASK_TYPE

        navi_bundle = _NAVI_BUNDLES[classifier]
        libraries = navi_bundle.libraries

        parameters = self._common_parameters(classifier, libraries, platform,
                                             ios_dynamic=navi_bundle.ios_build_dynamic,
                                             ios_mangle_symbols=navi_bundle.ios_mangle_symbols)
        parameters.update(self._navi_parameters())
        self._export(subtask_type, parameters, classifier)

    def _export(self, subtask_type, parameters, classifier):
        SubtaskRunner(self).run(
            subtask_type,
            parameters,
            tag='{}_{}'.format(classifier, parameters['platform'] or 'dev'),
            wait=False,
            description='{version} {platform} bundle with libraries=[{libraries}]'.format(
                        version=parameters['custom_release_version'],
                        platform=(parameters['platform'] or 'dev'),
                        libraries=parameters['libraries']
                        ),
            )

    def _common_parameters(self, classifier, libraries, platform, platform_libraries=None, ios_dynamic=None, ios_mangle_symbols=None):
        parameters = {
            'arcadia_url': self.Context.arcadia_url,
            'custom_release_version': '{}-{}'.format(self.Parameters.branch_revision, classifier),
            'libraries': libraries,
            'platform': platform
        }
        if platform == 'ios' or platform == 'android':
            if platform_libraries:
                parameters.update({
                    'platform_libraries': platform_libraries
                })
        if platform == 'ios':
            assert ios_dynamic is not None and ios_mangle_symbols is not None
            parameters.update({
                'is_ios_dynamic': ios_dynamic,
                'mangle_symbols': ios_mangle_symbols
            })
        return parameters

    def _simple_parameters(self):
        return {
            'generate_doc': True,
            'is_navi': False,
            'is_snapshot': False,
        }

    def _navi_parameters(self):
        return {
            'is_navi': True,
            'is_upload': True,
        }
